import Jensen, { DeviceInfo, ScheduleInfo } from "../../";
import { Logger } from "../Logger";

// 指令参数
export type CommandOptions = {
    exclusive?: boolean;        // 是否排它性的，在排它性的事务执行期间，其它所有的新事务都不再等待而是直接返回错误
    expires?: number;           // 超时时长
    retry?: number;             // 重试次数
    wait?: number;              // 等待前一个事务完结的最大时长
    reconnect?: boolean;        // 是否重连来确保事务的成功
    params?: any[];             // 原方法的参数
}

// 指令代理函数
export type CommandProxy = (jensen: Jensen | null) => Promise<any> | undefined;

// 指令请求
export type CommandRequest = {
    id: string;
    command: CommandProxy;
    options: CommandOptions;
    createTime: number;
}

export type CommandPromise = {
    id: string;
    resolve: (value:any) => void;
    reject: (error:any) => void;
    timeout: number;
}

// 定时任务
export type ScheduledTask = {
    interval: number;
    command: CommandProxy;
    onComplete?: (data: any) => void;
}

// 定时任务信息，增加了执行结果的记录
type ScheduledTaskInfo = {
    interval: number;
    lastExecuteTime: number;
    executedCount: number;
    command: CommandProxy;
    onComplete?: (data: any) => void;
}

type AutoConnectEventHandler = (dinfo: DeviceInfo) => void;
type ConnectionEventHandler = (state: ConnectionStatus, dinfo: DeviceInfo | string | null) => void;

// 连接状态枚举
export type ConnectionStatus = 'connect-timeout' | 'init' | 'connected' | 'reconnected' | 'disconnected';

// exec('set-time', { exclusive: false, expires: 5, })
// setTime(false, true, new Date(), 5)
// 

// 连接管理
export class ConnectionManager
{
    private logger: typeof Logger;
    private connections: Map<string, Jensen>;
    private onautoconnectEventListener: AutoConnectEventHandler | null = null;
    private onconnectionstatechangedListener: ConnectionEventHandler | null = null;
    constructor (_logger: typeof Logger) {
        this.logger = _logger;
        this.connections = new Map<string, Jensen>();
        this.registerUSBEventListener();
    }

    async registerUSBEventListener () {
        const usb = (navigator as any).usb;
        const self = this;
        usb.onconnect = (evt: any) => {
            try
            {
                let dev = evt.device;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) return;
                self._connect(dev);
            }
            catch(ex)
            {
                this.logger.error('jensen', 'onconnect', String(ex));
            }
        }
        usb.ondisconnect = (evt: any) => {
            try
            {
                let dev = evt.device;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) return;
                self._disconnect(dev);
            }
            catch(ex)
            {
                this.logger.error('jensen', 'ondisconnect', String(ex));
            }
        }
    }

    // 尝试连接，遍历所有的WebUSB设备，如果是HiDock设备，则直接建立连接，并且还要
    async tryconnect () {
        const usb = (navigator as any).usb;
        const self = this;
        usb.getDevices().then(async (devices: any[]) => {
            for (let i = 0; i < devices.length; i++)
            {
                let dev = devices[i];
                if (dev.opend) continue;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) continue;
                // console.log('auto connect', dev);
                self.logger.info('jensen', 'tryconnect', 'VID: ' + dev.vendorId + ', PID: ' + dev.productId);
                // 使用dev创建新的Jensen实例
                try
                {
                    self._connect(dev);
                }
                catch(ex)
                {
                    self.logger.error('jensen', 'tryconnect', String(ex));
                }
            }
        });
    }

    // 连接事件
    private async _onstatechanged (state: ConnectionStatus, jensen: Jensen) {

    }

    // 当设备断开连接时触发，如果设备是之前已经连接过的，那还需要触发ondisconnect事件
    private async _disconnect(dev: any) {
        let self = this;
        this.connections.forEach((j, k) => {
            let ud = j.getUSBDevice();
            if (ud == dev)
            {
                // console.log(k + ' disconnected...');
                self.logger.info('jensen', 'ondisconnect', k + ' disconnected');
                self.onconnectionstatechangedListener?.('disconnected', k);
            }
        });
    }

    // 当设备连接时触发，如果设备是之前已经连接过的，那还需要触发onconnect事件
    private async _connect (dev: any) {
        await dev.open();
        let inst = new Jensen(Logger, dev);
        this.logger.info('jensen', 'auto-connect', 'initialize');
        await inst.initialize();
        this.onconnectionstatechangedListener?.('init', inst.getModel());
        let dinfo: DeviceInfo | null = null;
        for (let i = 0; i < 30; i++)
        {
            try
            {
                console.error('dinfo', 'before', new Date().toLocaleString());
                await sleep(2000);
                dinfo = await inst.getDeviceInfo(5);
                console.error('dinfo', dinfo, new Date().toLocaleString());
                if (dinfo == null)
                {
                    inst.reconnect();
                    await sleep(2000);
                    continue;
                }
                break;
            }
            catch(e)
            {
                // ...
            }
        }
        if(dinfo == null) {
            console.log('dinfo is null', dev);
        }
        if(dinfo == null && dev.opened) {
            this.onconnectionstatechangedListener?.('connect-timeout', inst.getModel());
            try {
                dev.close();
                dev.forget();
            } catch (error) {
                // do nothing
            }
            return;
        }
        this.logger.info('jensen', 'auto-connect', JSON.stringify(dinfo));
        if (dinfo)
        {
            // 如果是之前已连接的设备，那就需要触发onconnect事件
            if (this.connections.has(dinfo.sn))
            {
                this.connections.set(
                    dinfo.sn, 
                    // new CommandManager(dinfo, inst)
                    inst
                );
                this.logger.info('jensen', 'onconnect', dinfo.sn + ' reconnected');
                // let jensen = this.connections.get(dinfo.sn);
                // jensen?.setUSBDevice(dev);
                this.onconnectionstatechangedListener?.('reconnected', dinfo);
                return;
            }
            this.connections.set(
                dinfo.sn, 
                // new CommandManager(dinfo, inst)
                inst
            );
            try
            {
                this.onautoconnectEventListener?.(dinfo);
                this.onconnectionstatechangedListener?.('connected', dinfo);
            }
            catch(err)
            {
                this.logger.error('jensen', 'autoconnect', String(err));
            }
        }
    }

    onautoconnect (eventListener: AutoConnectEventHandler) {
        this.onautoconnectEventListener = eventListener;
    }

    // 这个方法用于管理 连接/断开连接 的回调
    onconnectionstatechanged (eventHandler: ConnectionEventHandler)
    {
        this.onconnectionstatechangedListener = eventHandler;
    }

    // 获取一个Jensen实例
    getInstance(tag: string) {
        return this.connections.get(tag);
    }
    // 设置一个Jensen实例
    setInstance(tag: string, jensen: Jensen, oldTag: string | null = null) {
        if (oldTag) this.connections.delete(oldTag);
        this.connections.set(tag, jensen);
    }

    // 主动连接新设备，如果没有连接成功或是未连接，则直接返回null，如果用户选择的是已连接的设备，则直接抛出异常
    // 但是异常的表现形式需要另外商量沟通
    async connect () {
        const usb = (navigator as any).usb;
        let self = this;
        let conn = await usb.requestDevice({
            filters: [{ vendorId: 0x10d6 }, { vendorId: 0x3887 }]
        });
        if (conn == null) return null;
        if (conn.opened) return Logger.error('jensen', 'connect', 'device already opened');
        await conn.open();
        let jensen = new Jensen(self.logger, conn);
        jensen.onerror = (err) => {
            self.logger.error('jensen', 'connect error', String(err));
            // myMessage.error(i18n.t('connection.error'), 10000);
        }
        await jensen.initialize();
        let dinfo = await jensen.getDeviceInfo();
        // @ts-ignore
        jensen.onerror = null;
        if (dinfo)
        {
            self.connections.set(
                dinfo.sn, 
                // new CommandManager(dinfo, inst)
                jensen
            );
            return dinfo
        }
        return null;
    }

    // 关闭连接
    async close (sn: string) {
        let jensen = this.getInstance(sn);
        if (jensen) jensen.disconnect();
        this.connections.delete(sn);
    }
}

// 指令管理/代理
class CommandManager
{
    // 设备信息
    private serialNumber: string | null = null;
    // 命令队列
    private commands:CommandRequest[] = [];
    // 指令Promise响应
    private directives: Map<string, CommandPromise> = new Map<string, CommandPromise>();
    // 定时任务
    private tasks: ScheduledTaskInfo[] = [];
    // 是否暂停定时任务
    private suspendTimer: boolean = false;
    // Jensen实例
    private jensen: Jensen | null = null;
    // 当前事务
    private currentTask: String | null = null;
    constructor(dinfo: DeviceInfo, jensen: Jensen)
    {
        // TODO: 要不要完全保存deviceInfo？
        this.serialNumber = dinfo.sn;
        this.jensen = jensen;
        // 开启定时器
        let self = this;
        window.setInterval(function() {
            self._executeTimer();
        }, 1000);
    }

    dump () {
        // TODO: 完成内部成员的状态输出，用于跟踪连接和事务状态
        console.log('###############################################################');
        console.log('SerialNumber', this.serialNumber);
        console.log('Tasks', this.tasks);
        console.log('Commands', this.commands);
        console.log('SuspendTimer', this.suspendTimer);
        console.log('Jensen Instance', this.jensen);
    }

    setup () {
        // 在连接初次建立时需要做的初始化工作，是不是放在这里还没有想好
    }

    // onconnect事件如何触发？
    // 是不是应该全局唯一触发？好像至少多少时间上就应该重新触发一次
    // 或者是每次触发，然后可以被别的地方接管走
    // 另外，还有一个问题，如果断开了连接再次连接上来，会话都不同了，还是要按SN做标识，否则多连接管理会出问题

    // 增加一个新指令消息
    // 怎么样触发新指令呢？
    async getDeviceInfo(secs?: number) {
        // 当前是否可以执行？
        // 我是不是要分一下组？
        // 我是不是要加一个模式转换的方法？
        return this.jensen?.getDeviceInfo(secs);
        // return this.request((jensen) => jensen?.getDeviceInfo(), { reconnect: false });
    }

    getSerialNumber () {
        return this.serialNumber;
    }

    // 更改模式，要不要超时时长？在超过多长时间后要销毁或者说是恢复过去？
    // 1. 当前是什么模式
    // 2. 请求指令的是来自于哪个模式？
    // 3. 整一个事务控制吧

    // 事务开始或结束，不传参数就是事务结束了
    // 对于每一个事务而言，应该要尽量确保它执行的成功的，包括重连尝试
    async begin(secs: number = 5) {
        // 如果应用端尝试变更事务切换不过去，那就应该一直等
        // 这里是不是要弄一个可以打断的？如果事务与当前事务一致，又应该怎么办？还是要阻止吧？
        // 结束事务
        if (this.currentTask)
        {
            throw new Error("pending task: " + this.currentTask);
        }
        let tid = this.uuid();
        this.currentTask = tid;
        return tid;
    }

    end (id: string) {
        if (this.currentTask != id) throw new Error('invalid task: ' + this.currentTask + ' --> ' + id);
        this.currentTask = null;
    }

    // 开始/停止定时任务
    // 但是如果停止的话，也只是停止下一轮的执行，要不要等待全部停止完成再说？
    startTimer () {
        this.suspendTimer = false;
    }
    stopTimer() {
        this.suspendTimer = true;
    }

    private async _executeTimer () {
        if (this.suspendTimer) return;
        // 遍历所有的定时器任务
        for (let i = 0; i < this.tasks.length; i++)
        {
            let item = this.tasks[i];
            let now = new Date().getTime() / 1000;
            if (now - item.lastExecuteTime < item.interval) continue;
            try
            {
                let rst = await item.command.apply(null, [this.jensen]);
                item.onComplete?.apply(null, [rst]);
                /*
                item.command.apply(null, [this.jensen])?.then((rst) => {
                    item.onComplete?.apply(null, [ rst ]);
                });
                */
            }
            catch(ex)
            {
                console.error(ex);
            }
            finally
            {
                item.lastExecuteTime = now;
            }
        }
    }

    // 注册定时任务，返回值为它在列表中的索引位置，可能后面有要解除定时执行的需要，先预留一下
    // 这里有个问题，我要如何完成回调呢？
    async registerTimer(task: ScheduledTask) {
        let len = this.tasks.length;
        this.tasks.push({
            interval: task.interval,
            command: task.command,
            lastExecuteTime: 0,
            executedCount: 0
        });
        return len;
    }

    // 指令分两种，有复杂结构的返回值的或是普通boolean类型的两种
    async request (func: CommandProxy, opts: CommandOptions) {
        // TODO: 需要在这里注册一个Promise
        let self = this;
        let rid = this.uuid();
        let future = new Promise((resolve, reject) => {
            self.directives.set(rid, {
                id: rid,
                resolve: resolve,
                reject: reject,
                timeout: opts.expires || 5
            });
        });
        this.commands.push({ command: func, id: rid, options: opts, createTime: new Date().getTime() });
        return future;
    }
    
    // 需要怎么样持续的执行呢？
    async execute(func: CommandProxy, opts: CommandOptions) {
        let tid = await this.begin(opts.wait);
        try
        {
            let args: any[] | undefined = opts.params;

            let execCount = 0;
            let maxTries = (opts.retry || 0) + 1;
            do {
                execCount += 1;
                try
                {
                    let rst: any = await func(this.jensen);
                    if (rst) return rst;
                    if (opts.reconnect)
                    {
                        // TODO: 连接重新建立
                        this.logger.info('jensen', 'execute', 'try reconnect');
                    }
                }
                catch(ex)
                {
                    console.error(ex);
                }
            }
            while (execCount < maxTries);
        }
        finally
        {
            this.end(tid);
        }
    }

    private async tryReconnect()
    {
        try
        {
            // TODO: 需要准备一个新的reconnect方法，使用同一个WebUSB实例来close/open来实现重连，否则必然会乱掉
            this.jensen?.reconnect();
        }
        catch(ex)
        {
            console.error(ex);
        }
    }

    async getTime (timeout?: number) {
        return this.jensen?.getTime(timeout);
    }

    async setTime (date: Date, timeout?: number) {
        return this.jensen?.setTime(date, timeout);
    }

    async getRecordingFile () {
        return this.jensen?.getRecordingFile();
    }

    async getFileCount (timeout?: number) {
        return this.jensen?.getFileCount(timeout);
    }

    async listFiles (timeout?: number) {
        return this.jensen?.listFiles(timeout);
    }

    async readFile (fname: string, offset: number, length: number) {
        return this.jensen?.readFile(fname, offset, length);
    }

    async getFile (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void, onprogress?: (size: number) => void) {
        return this.jensen?.getFile(fileName, length, on, onprogress);
    }

    async getFilePart (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void, onprogress?: (size: number) => void) {
        return this.jensen?.getFilePart(fileName, length, on, onprogress);
    }

    async getFileBlock (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void) {
        return this.jensen?.getFileBlock(fileName, length, on);
    }

    async requestFirmwareUpgrade (vn: number, length: number, timeout?: number) {
        return this.jensen?.requestFirmwareUpgrade(vn, length, timeout);
    }

    async uploadFirmware (data: number[], timeout?: number, onProgress?: (cur: number, total: number) => void) {
        return this.jensen?.uploadFirmware(data, timeout, onProgress);
    }

    async beginBNC (timeout?: number) {
        return this.jensen?.beginBNC(timeout);
    }

    async endBNC (timeout?: number) {
        return this.jensen?.endBNC(timeout);
    }

    async deleteFile (fileName: string) {
        return this.jensen?.deleteFile(fileName);
    }

    async getSettings (timeout?: number) {
        return this.jensen?.getSettings(timeout);
    }
    
    async setAutoRecord (enable: boolean, timeout?: number) {
        return this.jensen?.setAutoRecord(enable, timeout);
    }

    async setAutoPlay (enable: boolean, timeout?: number) {
        return this.jensen?.setAutoPlay(enable, timeout);
    }

    async setNotification (enable: boolean, timeout?: number) {
        return this.jensen?.setNotification(enable, timeout);
    }

    async setBluetoothPromptPlay (enable: boolean, timeout?: number) {
        return this.jensen?.setBluetoothPromptPlay(enable, timeout);
    }

    async getCardInfo (timeout?: number) {
        return this.jensen?.getCardInfo(timeout);
    }

    async formatCard (timeout?: number) {
        return this.jensen?.formatCard(timeout);
    }

    async factoryReset (timeout?: number) {
        return this.jensen?.factoryReset(timeout);
    }

    async restoreFactorySettings (timeout?: number) {
        return this.jensen?.restoreFactorySettings(timeout);
    }

    async recordTestStart (type: number, timeout?: number) {
        return this.jensen?.recordTestStart(type, timeout);
    }

    async recordTestEnd (type: number, timeout?: number) {
        return this.jensen?.recordTestEnd(type, timeout);
    }

    async test (timeout?: number) {
        return this.jensen?.test(timeout);
    }

    async writeSerialNumber (sn: string) {
        return this.jensen?.writeSerialNumber(sn);
    }

    async getRealtimeSettings () {
        return this.jensen?.getRealtimeSettings();
    }

    async startRealtime () {
        return this.jensen?.startRealtime();
    }

    async pauseRealtime () {
        return this.jensen?.pauseRealtime();
    }

    async stopRealtime () {
        return this.jensen?.stopRealtime();
    }
    
    async getRealtime (frames: number) {
        return this.jensen?.getRealtime(frames);
    }

    async scanDevices (timeout?: number) {
        return this.jensen?.scanDevices(timeout);
    }
    
    async connectBTDevice (mac: string, timeout?: number) {
        return this.jensen?.connectBTDevice(mac, timeout);
    }

    async disconnectBTDevice (timeout?: number) {
        return this.jensen?.disconnectBTDevice(timeout);
    }

    async getBluetoothStatus (timeout?: number) {
        return this.jensen?.getBluetoothStatus(timeout);
    }

    async requestToneUpdate (signature: string, size: number, timeout?: number) {
        return this.jensen?.requestToneUpdate(signature, size, timeout);
    }
    
    async updateTone (data: Uint8Array, timeout?: number) {
        return this.jensen?.updateTone(data, timeout);
    }
    
    async requestUACUpdate (signature: string, size: number, timeout?: number) {
        return this.jensen?.requestUACUpdate(signature, size, timeout);
    }

    async updateUAC (data: Uint8Array, timeout?: number) {
        return this.jensen?.updateUAC(data, timeout);
    }

    async sendScheduleInfo (info: ScheduleInfo[]) {
        return this.jensen?.sendScheduleInfo(info);
    }

    // 生成唯一id，暂时用随机数生成，后面最好用顺序递增的整数比较好
    private uuid () {
        return String(Math.floor(1000_0000_0000_0000 + Math.random() * 1000_0000_0000_0000));
    }
}

function sleep(ms: number) {
    return new Promise(resolve => setTimeout(resolve, ms));
}


export const mgr = new ConnectionManager(Logger);