import Jensen, { BatteryStatus, BluetoothStatus, DeviceInfo, FileInfo } from "../../jensen";
import { Logger } from "../Logger";

// ===================== 设备状态存储相关类型 =====================

// 状态变更监听器类型
export type StateChangeListener<T> = (key: string, newValue: T, oldValue: T | undefined) => void;

// 设备状态Schema - 定义所有可存储的状态类型
export interface DeviceStateSchema {
    deviceInfo: DeviceInfo | null;
    time: { deviceTime: string; syncTime: Date } | null;
    settings: { autoRecord: boolean; autoPlay: boolean; notification?: boolean } | null;
    cardInfo: { used: number; capacity: number; status: string } | null;
    batteryStatus: BatteryStatus | null;
    bluetoothStatus: BluetoothStatus | null;
    files: FileInfo[] | null;
    // 可按需扩展其他状态...
    [key: string]: any;  // 允许动态扩展
}

// ===================== 设备状态存储类 =====================

export class DeviceStateStore {
    // 内部数据存储
    private state: Partial<DeviceStateSchema> = {};
    // 更新时间记录
    private updateTimes: Map<string, number> = new Map();
    // 状态变更监听器
    private listeners: Map<string, StateChangeListener<any>[]> = new Map();

    // 获取状态
    get<K extends keyof DeviceStateSchema>(key: K): DeviceStateSchema[K] | undefined {
        return this.state[key];
    }

    // 设置状态
    set<K extends keyof DeviceStateSchema>(key: K, value: DeviceStateSchema[K]): void {
        const oldValue = this.state[key];
        this.state[key] = value;
        this.updateTimes.set(key as string, Date.now());
        // 触发监听器
        this.notifyListeners(key as string, value, oldValue);
    }

    // 获取状态最后更新时间
    getUpdateTime(key: keyof DeviceStateSchema): number | undefined {
        return this.updateTimes.get(key as string);
    }

    // 判断状态是否过期
    isStale(key: keyof DeviceStateSchema, maxAge: number): boolean {
        const updateTime = this.updateTimes.get(key as string);
        if (!updateTime) return true;
        return Date.now() - updateTime > maxAge;
    }

    // 检查状态是否存在
    has(key: keyof DeviceStateSchema): boolean {
        return key in this.state && this.state[key] !== undefined;
    }

    // 订阅状态变更
    subscribe<K extends keyof DeviceStateSchema>(
        key: K,
        listener: StateChangeListener<DeviceStateSchema[K]>
    ): () => void {
        const keyStr = key as string;
        if (!this.listeners.has(keyStr)) {
            this.listeners.set(keyStr, []);
        }
        this.listeners.get(keyStr)!.push(listener);

        // 返回取消订阅函数
        return () => {
            const arr = this.listeners.get(keyStr);
            if (arr) {
                const idx = arr.indexOf(listener);
                if (idx !== -1) arr.splice(idx, 1);
            }
        };
    }

    // 触发监听器
    private notifyListeners<T>(key: string, newValue: T, oldValue: T | undefined): void {
        const arr = this.listeners.get(key);
        if (arr) {
            arr.forEach(listener => {
                try {
                    listener(key, newValue, oldValue);
                } catch (e) {
                    console.error('StateChangeListener error:', e);
                }
            });
        }
    }

    // 清空所有状态
    clear(): void {
        this.state = {};
        this.updateTimes.clear();
        // 不清除监听器，允许重连后继续监听
    }

    // 清空所有监听器
    clearListeners(): void {
        this.listeners.clear();
    }

    // 导出状态快照（用于调试）
    dump(): { state: Partial<DeviceStateSchema>; updateTimes: Record<string, number> } {
        const times: Record<string, number> = {};
        this.updateTimes.forEach((v, k) => times[k] = v);
        return {
            state: { ...this.state },
            updateTimes: times
        };
    }
}

// ===================== 任务类型定义 =====================

export type InitializeTask = {
    tag: string;
    task: (jensen: Jensen | null, store: DeviceStateStore) => Promise<any>;
}

export type TimerTask = {
    tag: string;
    interval: number;
    lastExecuteTime?: number;
    task: (jensen: Jensen | null, store: DeviceStateStore) => Promise<any>;
    onComplete?: (success: boolean, data: any) => void;
}

export type BackgroundTask = {
    tag: string;
    // 用于任务调度
    schedule: (jensen: Jensen | null, store: DeviceStateStore) => Promise<any>;
    // 用于设备交互使用
    deviceTask?: (jensen: Jensen | null, store: DeviceStateStore) => Promise<any>;
    // 任务善后处理调用
    generalTask?: (jensen: Jensen | null, store: DeviceStateStore) => Promise<any>;
    lastExecuteTime: number;
    interval: number;  // 轮询间隔
}

export type UserTask = {
    tag: string;
    startTime: number;
    lastActiveTime: number;
}

// 用户任务最大空闲时间，超过这个时间则直接取消任务执行权限
const USER_TASK_EXPIRE_TIME = 30 * 1000;

export class TaskManager {
    private logger: typeof Logger;
    private jensen: Jensen | null;

    // 是否已经启动了任务调度
    private started: boolean = false;
    private runnable: boolean = true;

    // 初始化任务
    private initializeTasks: InitializeTask[] = [];

    // 定时任务
    private timerTasks: TimerTask[] = [];

    // 后台任务
    private backgroundTasks: BackgroundTask[] = [];

    // 当前用户任务，排它性的任务
    private userTask: UserTask | null = null;

    // 设备状态存储
    private stateStore: DeviceStateStore = new DeviceStateStore();

    constructor (jensen: Jensen | null, _logger: typeof Logger)
    {
        this.jensen = jensen;
        this.logger = _logger;
    }

    // 获取状态存储实例（供外部访问）
    getStateStore(): DeviceStateStore {
        return this.stateStore;
    }

    // 快捷方法：获取状态
    getState<K extends keyof DeviceStateSchema>(key: K): DeviceStateSchema[K] | undefined {
        return this.stateStore.get(key);
    }

    // 快捷方法：设置状态
    setState<K extends keyof DeviceStateSchema>(key: K, value: DeviceStateSchema[K]): void {
        this.stateStore.set(key, value);
    }

    // 快捷方法：订阅状态变更
    subscribeState<K extends keyof DeviceStateSchema>(
        key: K,
        listener: StateChangeListener<DeviceStateSchema[K]>
    ): () => void {
        return this.stateStore.subscribe(key, listener);
    }

    // 注册初始化任务
    registerInitializeTask(task: InitializeTask) {
        this.initializeTasks.push(task);
    }

    // 注册定时任务
    registerTimerTask(task: TimerTask) {
        this.timerTasks.push(task);
    }

    // 注册后台任务
    registerBackgroundTask(task: BackgroundTask) {
        this.backgroundTasks.push(task);
    }

    // 申请用户任务，当timeout后仍然没有申请到，则返回false
    async applyUserTask(tag: string, timeout: number = 3000) {
        const stime = new Date().getTime();
        while (true) {
            if (new Date().getTime() - stime > timeout) return false;
            if (this.userTask)
            {
                const now = new Date().getTime();
                if (now - this.userTask.lastActiveTime > USER_TASK_EXPIRE_TIME)
                {
                    this.userTask = null;
                    break;
                }
                await sleep(1000);
                continue;
            }
            break;
        }
        const now = new Date().getTime();
        this.userTask = {
            tag: tag,
            startTime: now,
            lastActiveTime: now
        };
        return true;
    }

    // 释放用户任务
    releaseUserTask(tag: string) {
        if (this.userTask?.tag === tag)
        {
            this.userTask = null;
            return true;
        }
        throw new Error('no task registered: ' + tag);
    }

    // 保持用户任务活跃（防止超时被释放）
    keepUserTaskAlive(tag: string)
    {
        if (this.userTask)
        {
            this.userTask.lastActiveTime = new Date().getTime();
        }
        else throw new Error('task not found: ' + tag);
    }

    // 验证用户任务是否有效
    // 返回 true 表示验证通过，否则抛出错误
    verifyUserTask(tag: string): boolean {
        if (!this.userTask) {
            throw new Error('No user task registered. Please register a user task first.');
        }
        if (this.userTask.tag !== tag) {
            throw new Error(`User task tag mismatch: expected "${this.userTask.tag}", got "${tag}"`);
        }
        // 检查是否过期
        const now = new Date().getTime();
        if (now - this.userTask.lastActiveTime > USER_TASK_EXPIRE_TIME) {
            this.userTask = null;
            throw new Error('User task has expired. Please register a new user task.');
        }
        // 更新活跃时间
        this.userTask.lastActiveTime = now;
        return true;
    }

    // 获取当前用户任务的 tag（如果存在）
    getCurrentUserTaskTag(): string | null {
        return this.userTask?.tag || null;
    }

    shutdown() {
        this.runnable = false;
        this.stateStore.clear();
        this.stateStore.clearListeners();
    }

    async scheduleInitializeTasks() {
        // TODO: 需要确保其它任务处于暂停状态
        for (let i = 0; i < this.initializeTasks.length; i++)
        {
            try
            {
                await this.initializeTasks[i].task(this.jensen, this.stateStore);
            }
            catch(ex)
            {
                Logger.error('jensen', 'initialize', String(ex));
            }
        }
    }

    // 任务的调度
    async scheduleTasks() {

        // 轮询执行定时任务和后台任务
        while (this.runnable)
        {
            // 如果用户任务活动超时，则释放用户任务
            if (this.userTask)
            {
                const now = new Date().getTime();
                if (now - this.userTask.lastActiveTime > USER_TASK_EXPIRE_TIME)
                {
                    Logger.error('mgr', 'task-manager', 'User task has expired: ' + this.userTask.tag);
                    this.userTask = null;
                }
            }

            // 遍历所有的后台任务，是否有执行的必要？
            for (let i = 0; i < this.backgroundTasks.length; i++)
            {
                let now = new Date().getTime();
                let item = this.backgroundTasks[i];
                if (this.userTask)
                {
                    item.lastExecuteTime = now;
                    continue;
                }
                try
                {
                    await item.schedule(this.jensen, this.stateStore);
                    item.lastExecuteTime = new Date().getTime();
                }
                catch(ex)
                {
                    Logger.error('jensen', 'bg-task', String(ex));
                }
            }

            // 遍历所有的定时任务，检查是否到了执行时间
            for (let i = 0; i < this.timerTasks.length; i++)
            {
                let now = new Date().getTime();
                let item = this.timerTasks[i];
                if (this.userTask)
                {
                    item.lastExecuteTime = now;
                    continue;
                }
                if (now - (item.lastExecuteTime || 0) < item.interval) continue;
                try
                {
                    await item.task(this.jensen, this.stateStore);
                    item.lastExecuteTime = new Date().getTime();
                }
                catch(ex)
                {
                    Logger.error('jensen', 'timer-task', String(ex));
                }
            }

            await sleep(100);
        }
    }

    // 导出状态快照（用于调试）
    dumpState() {
        return this.stateStore.dump();
    }
}

function sleep(time: number) {
    return new Promise(resolve => setTimeout(resolve, time));
}
