import Jensen, { DeviceInfo, ScheduleInfo } from "../../";
import { Logger } from "../Logger";
import { BackgroundTask, DeviceStateSchema, DeviceStateStore, InitializeTask, StateChangeListener, TaskManager, TimerTask } from "./tasks";

// 指令参数
export type CommandOptions = {
    exclusive?: boolean;        // 是否排它性的，在排它性的事务执行期间，其它所有的新事务都不再等待而是直接返回错误
    expires?: number;           // 超时时长
    retry?: number;             // 重试次数
    wait?: number;              // 等待前一个事务完结的最大时长
    reconnect?: boolean;        // 是否重连来确保事务的成功
    params?: any[];             // 原方法的参数
}

// 指令代理函数
export type CommandProxy = (jensen: Jensen | null) => Promise<any> | undefined;

// 指令请求
export type CommandRequest = {
    id: string;
    command: CommandProxy;
    options: CommandOptions;
    createTime: number;
}

export type CommandPromise = {
    id: string;
    resolve: (value:any) => void;
    reject: (error:any) => void;
    timeout: number;
}

// 定时任务
export type ScheduledTask = {
    interval: number;
    command: CommandProxy;
    onComplete?: (data: any) => void;
}

// 定时任务信息，增加了执行结果的记录
type ScheduledTaskInfo = {
    interval: number;
    lastExecuteTime: number;
    executedCount: number;
    command: CommandProxy;
    onComplete?: (data: any) => void;
}

type AutoConnectEventHandler = (dinfo: DeviceInfo) => void;
type ConnectionEventHandler = (state: ConnectionStatus, dinfo: DeviceInfo | string | null) => void;

// 连接状态枚举
export type ConnectionStatus = 'connect-timeout' | 'init' | 'connected' | 'reconnected' | 'disconnected';

// exec('set-time', { exclusive: false, expires: 5, })
// setTime(false, true, new Date(), 5)
// 

// 连接管理
export class ConnectionManager
{
    private logger: typeof Logger;
    private connections: Map<string, Jensen>;
    private onautoconnectEventListener: AutoConnectEventHandler | null = null;
    private onconnectionstatechangedListener: ConnectionEventHandler | null = null;

    constructor (_logger: typeof Logger) {
        this.logger = _logger;
        this.connections = new Map<string, Jensen>();
        this.registerUSBEventListener();
    }

    // 任务管理器，按设备分组管理
    private taskManagers: Map<string, TaskManager> = new Map<string, TaskManager>();

    // 注册初始化事务
    async registerInitializeTask (sn: string, task: InitializeTask) {
        return this.taskManagers.get(sn)?.registerInitializeTask(task);
    }

    // 注册定时器任务
    async registerTimerTask (sn: string, task: TimerTask) {
        return this.taskManagers.get(sn)?.registerTimerTask(task);
    }

    // 注册后台任务
    async registerBackgroundTask (sn: string, task: BackgroundTask) {
        return this.taskManagers.get(sn)?.registerBackgroundTask(task);
    }

    // 注册用户任务
    async registerTask (sn: string, tag: string) {
        return this.taskManagers.get(sn)?.applyUserTask(tag);
    }

    // 注销任务，任意类型的任务都可以注销
    async unregisterTask (sn: string, tag: string) {
        return this.taskManagers.get(sn)?.releaseUserTask(tag);
    }

    // 用户任务保活
    async keepUserTaskAlive (sn: string, tag: string) {
        return this.taskManagers.get(sn)?.keepUserTaskAlive(tag);
    }

    // 验证用户任务
    verifyUserTask (sn: string, tag: string): boolean {
        const tm = this.taskManagers.get(sn);
        if (!tm) throw new Error('Device not found: ' + sn);
        return tm.verifyUserTask(tag);
    }

    // 获取当前用户任务的 tag
    getCurrentUserTaskTag (sn: string): string | null {
        return this.taskManagers.get(sn)?.getCurrentUserTaskTag() || null;
    }

    // 安全执行：验证用户任务后执行回调
    // 使用方法: await mgr.executeWithUserTask(sn, tag, (jensen) => jensen.getTime(1));
    async executeWithUserTask<T> (
        sn: string, 
        tag: string, 
        callback: (jensen: Jensen) => Promise<T>
    ): Promise<T> {
        // 先验证用户任务
        this.verifyUserTask(sn, tag);
        // 获取原始 jensen 实例（内部使用，已验证过 UserTask）
        const jensen = this.getInstanceRaw(sn);
        if (!jensen) throw new Error('Device not connected: ' + sn);
        // 执行回调
        return await callback(jensen);
    }

    async registerUSBEventListener () {
        const usb = (navigator as any).usb;
        const self = this;
        usb.onconnect = (evt: any) => {
            try
            {
                let dev = evt.device;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) return;
                self._connect(dev);
            }
            catch(ex)
            {
                this.logger.error('jensen', 'onconnect', String(ex));
            }
        }
        usb.ondisconnect = (evt: any) => {
            try
            {
                let dev = evt.device;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) return;
                self._disconnect(dev);
            }
            catch(ex)
            {
                this.logger.error('jensen', 'ondisconnect', String(ex));
            }
        }
    }

    // 尝试连接，遍历所有的WebUSB设备，如果是HiDock设备，则直接建立连接，并且还要
    async tryconnect () {
        const usb = (navigator as any).usb;
        const self = this;
        usb.getDevices().then(async (devices: any[]) => {
            for (let i = 0; i < devices.length; i++)
            {
                let dev = devices[i];
                if (dev.opend) continue;
                if (dev.vendorId != 0x10d6 && dev.vendorId != 0x3887) continue;
                // console.log('auto connect', dev);
                self.logger.info('jensen', 'tryconnect', 'VID: ' + dev.vendorId + ', PID: ' + dev.productId);
                // 使用dev创建新的Jensen实例
                try
                {
                    self._connect(dev);
                }
                catch(ex)
                {
                    self.logger.error('jensen', 'tryconnect', String(ex));
                }
            }
        });
    }

    // 连接事件
    private async _onstatechanged (state: ConnectionStatus, jensen: Jensen) {

    }

    // 当设备断开连接时触发，如果设备是之前已经连接过的，那还需要触发ondisconnect事件
    private async _disconnect(dev: any) {
        let self = this;
        this.connections.forEach((j, k) => {
            let ud = j.getUSBDevice();
            if (ud == dev)
            {
                // console.log(k + ' disconnected...');
                self.logger.info('jensen', 'ondisconnect', k + ' disconnected');
                self.onconnectionstatechangedListener?.('disconnected', k);
                self.taskManagers.get(k)?.shutdown();
                self.taskManagers.delete(k);
            }
        });
    }

    // 当设备连接时触发，如果设备是之前已经连接过的，那还需要触发onconnect事件
    private async _connect (dev: any) {
        await dev.open();
        let inst = new Jensen(Logger, dev);
        this.logger.info('jensen', 'auto-connect', 'initialize');
        await inst.initialize();
        this.onconnectionstatechangedListener?.('init', inst.getModel());
        let dinfo: DeviceInfo | null = null;
        for (let i = 0; i < 30; i++)
        {
            try
            {
                console.error('dinfo', 'before', new Date().toLocaleString());
                await sleep(2000);
                dinfo = await inst.getDeviceInfo(5);
                console.error('dinfo', dinfo, new Date().toLocaleString());
                if (dinfo == null)
                {
                    inst.reconnect();
                    await sleep(2000);
                    continue;
                }
                break;
            }
            catch(e)
            {
                // ...
            }
        }
        if(dinfo == null) {
            console.log('dinfo is null', dev);
        }
        if(dinfo == null && dev.opened) {
            this.onconnectionstatechangedListener?.('connect-timeout', inst.getModel());
            try {
                dev.close();
                dev.forget();
            } catch (error) {
                // do nothing
            }
            return;
        }
        this.logger.info('jensen', 'auto-connect', JSON.stringify(dinfo));
        if (dinfo)
        {
            // 如果是之前已连接的设备，那就需要触发onconnect事件
            if (this.connections.has(dinfo.sn))
            {
                this.connections.set(
                    dinfo.sn, 
                    inst
                );
                this.logger.info('jensen', 'onconnect', dinfo.sn + ' reconnected');
                // let jensen = this.connections.get(dinfo.sn);
                // jensen?.setUSBDevice(dev);
                this.onconnectionstatechangedListener?.('reconnected', dinfo);
                return;
            }
            this.connections.set(
                dinfo.sn, 
                inst
            );
            // 新建TaskManager的时候，它的各类Tasks应该如何初始化？
            // 应该有一个统一的登记的地方
            let tm = this.taskManagers.get(dinfo.sn);
            if (tm == null) {
                tm = new TaskManager(inst, this.logger);
                this.taskBuilder(inst, tm);
                await tm.scheduleInitializeTasks();
                this.taskManagers.set(dinfo.sn, tm);
            }

            // 先触发连接事件，再启动定时任务循环
            try
            {
                this.onautoconnectEventListener?.(dinfo);
                this.onconnectionstatechangedListener?.('connected', dinfo);
            }
            catch(err)
            {
                this.logger.error('jensen', 'autoconnect', String(err));
            }

            // scheduleTasks 是一个无限循环，不要 await 它，否则后续代码永远不会执行
            tm.scheduleTasks();
        }
    }

    taskBuilder(jensen: Jensen, tm: TaskManager)
    {
        // 初始化任务的注册 - 数据会被存储到 DeviceStateStore 中
        tm.registerInitializeTask({
            tag: 'time-sync',
            task: async (jensen: Jensen | null, store) => {
                let time = await jensen?.getTime(1);
                Logger.info('jensen', 'initialize', 'get-time: ' + JSON.stringify(time));
                
                // 存储设备时间到状态容器
                if (time) {
                    store.set('time', { deviceTime: time.time, syncTime: new Date() });
                }
                
                // 同步设置时间
                await jensen?.setTime(new Date(), 1);
                Logger.info('jensen', 'initialize', 'set-time-to: ' + new Date().toLocaleString());
            }
        });

        tm.registerInitializeTask({
            tag: 'get-settings',
            task: async (jensen: Jensen | null, store) => {
                let settings = await jensen?.getSettings(1);
                Logger.info('jensen', 'initialize', 'get-settings: ' + JSON.stringify(settings));
                
                // 存储设置到状态容器
                if (settings) {
                    store.set('settings', settings);
                }
            }
        });

        tm.registerInitializeTask({
            tag: 'get-card-info',
            task: async (jensen: Jensen | null, store) => {
                let cardInfo = await jensen?.getCardInfo(1);
                Logger.info('jensen', 'initialize', 'get-card-info: ' + JSON.stringify(cardInfo));
                
                // 存储卡信息到状态容器
                if (cardInfo) {
                    store.set('cardInfo', cardInfo);
                }
            }
        });

        tm.registerInitializeTask({
            tag: 'battery-status',
            task: async (jensen: Jensen | null, store) => {
                if (jensen?.getModel() != 'hidock-p1') return;
                let batteryStatus = await jensen?.getBatteryStatus(1);
                Logger.info('jensen', 'initialize', 'get-battery-status: ' + JSON.stringify(batteryStatus));
                
                // 存储电池状态到状态容器
                if (batteryStatus) {
                    store.set('battery-status', batteryStatus);
                }
            }
        });

        // 定时同步设备电池状态
        tm.registerTimerTask({
            tag: 'battery-status',
            interval: 1000,
            task: async (jensen: Jensen | null, store) => {
                if (jensen?.getModel() != 'hidock-p1') return;
                let batteryStatus = await jensen?.getBatteryStatus(1);
                Logger.info('jensen', 'timer-task', 'battery: ' + JSON.stringify(batteryStatus));
                
                // 更新电池状态（会自动触发订阅者）
                if (batteryStatus) {
                    store.set('battery-status', batteryStatus);
                }
            }
        });

        // 定时获取设备时间
        tm.registerTimerTask({
            tag: 'get-device-time',
            interval: 1000,
            task: async (jensen: Jensen | null, store) => {
                if (jensen?.getModel() != 'hidock-p1') return;
                let time = await jensen?.getTime(1);
                Logger.info('jensen', 'timer-task', 'time: ' + JSON.stringify(time));
                
                // 更新设备时间
                if (time) {
                    store.set('time', { deviceTime: time.time, syncTime: new Date() });
                }
            }
        });

        // 录音中的状态同步
        tm.registerTimerTask({
            tag: 'recording-status',
            interval: 1000,
            task: async (jensen: Jensen | null, store) => {
                let recording = await jensen?.getRecordingFile();
                Logger.info('jensen', 'timer-task', 'recording-status: ' + JSON.stringify(recording));
                if (recording) store.set('recording', recording);
            }
        });

        // 设备录音文件列表同步
        tm.registerTimerTask({
            tag: 'file-list-sync',
            interval: 1000,
            task: async (jensen: Jensen | null, store) => {
                let needRefresh = false;
                // 1. 定时查询文件数量
                let fileCount = await jensen?.getFileCount(1);
                if (fileCount)
                {
                    let lastCount = store.get('file-count');
                    if (lastCount != fileCount.count)
                    {
                        needRefresh = true;
                        Logger.info('jensen', 'timer-task', 'file-count changed: ' + lastCount + ' -> ' + fileCount.count);
                    }
                    store.set('file-count', fileCount.count);
                }
                if (!needRefresh) return;
                let files = await jensen?.listFiles();
                if (files) store.set('files', files);
            }
        });

        // 后台任务的注册（如有需要可以在这里添加）
    }

    onautoconnect (eventListener: AutoConnectEventHandler) {
        this.onautoconnectEventListener = eventListener;
    }

    // 这个方法用于管理 连接/断开连接 的回调
    onconnectionstatechanged (eventHandler: ConnectionEventHandler)
    {
        this.onconnectionstatechangedListener = eventHandler;
    }

    // ===================== 设备状态访问接口 =====================

    // 获取指定设备的状态存储
    getDeviceStateStore(sn: string): DeviceStateStore | undefined {
        return this.taskManagers.get(sn)?.getStateStore();
    }

    // 获取指定设备的某个状态值
    getDeviceState<K extends keyof DeviceStateSchema>(sn: string, key: K): DeviceStateSchema[K] | undefined {
        return this.taskManagers.get(sn)?.getState(key);
    }

    // 设置指定设备的某个状态值
    setDeviceState<K extends keyof DeviceStateSchema>(sn: string, key: K, value: DeviceStateSchema[K]): void {
        this.taskManagers.get(sn)?.setState(key, value);
    }

    // 订阅指定设备的状态变更
    subscribeDeviceState<K extends keyof DeviceStateSchema>(
        sn: string,
        key: K,
        listener: StateChangeListener<DeviceStateSchema[K]>
    ): (() => void) | undefined {
        return this.taskManagers.get(sn)?.subscribeState(key, listener);
    }

    // 检查指定设备的状态是否过期
    isDeviceStateStale(sn: string, key: keyof DeviceStateSchema, maxAge: number): boolean {
        return this.taskManagers.get(sn)?.getStateStore().isStale(key, maxAge) ?? true;
    }

    // 导出指定设备的状态快照（用于调试）
    dumpDeviceState(sn: string) {
        return this.taskManagers.get(sn)?.dumpState();
    }

    // ===================== 原有接口 =====================

    // 获取一个受保护的 Jensen 实例（代理对象）
    // 所有方法调用都会自动验证 UserTask
    // 参数: sn - 设备序列号, userTaskTag - 用户任务标签
    getInstance(sn: string, userTaskTag?: string): Jensen | null {
        const jensen = this.connections.get(sn);
        if (!jensen) return null;
        
        // 如果没有提供 userTaskTag，返回只读代理，阻止所有方法调用
        // 如果提供了 userTaskTag，返回验证代理
        const taskManager = this.taskManagers.get(sn);
        const self = this;
        
        return new Proxy(jensen, {
            get(target, prop, receiver) {
                const value = Reflect.get(target, prop, receiver);
                
                // 如果是函数，则包装它以进行验证
                if (typeof value === 'function') {
                    return function(...args: any[]) {
                        // 验证 UserTask
                        if (!userTaskTag) {
                            throw new Error('getInstance() requires a userTaskTag parameter. Usage: mgr.getInstance(sn, userTaskTag)');
                        }
                        if (!taskManager) {
                            throw new Error('Device not found: ' + sn);
                        }
                        // 调用验证方法（会抛出错误如果验证失败）
                        taskManager.verifyUserTask(userTaskTag);
                        
                        // 验证通过，执行原方法
                        return value.apply(target, args);
                    };
                }
                
                // 非函数属性直接返回
                return value;
            }
        }) as Jensen;
    }

    // 获取原始的 Jensen 实例（仅供内部使用，不对外暴露）
    // 如果需要在内部任务中使用，可以调用此方法
    private getInstanceRaw(sn: string): Jensen | undefined {
        return this.connections.get(sn);
    }

    // 设置一个Jensen实例
    setInstance(sn: string, jensen: Jensen, oldTag: string | null = null) {
        if (oldTag) this.connections.delete(oldTag);
        this.connections.set(sn, jensen);
    }

    // 主动连接新设备，如果没有连接成功或是未连接，则直接返回null，如果用户选择的是已连接的设备，则直接抛出异常
    // 但是异常的表现形式需要另外商量沟通
    async connect () {
        const usb = (navigator as any).usb;
        let self = this;
        let conn = await usb.requestDevice({
            filters: [{ vendorId: 0x10d6 }, { vendorId: 0x3887 }]
        });
        if (conn == null) return null;
        if (conn.opened) return Logger.error('jensen', 'connect', 'device already opened');
        await conn.open();
        let jensen = new Jensen(self.logger, conn);
        jensen.onerror = (err) => {
            self.logger.error('jensen', 'connect error', String(err));
            // myMessage.error(i18n.t('connection.error'), 10000);
        }
        await jensen.initialize();
        let dinfo = await jensen.getDeviceInfo();
        // @ts-ignore
        jensen.onerror = null;
        if (dinfo)
        {
            self.connections.set(
                dinfo.sn, 
                jensen
            );
            return dinfo
        }
        return null;
    }

    // 关闭连接
    async close (sn: string) {
        let jensen = this.getInstanceRaw(sn);
        if (jensen) jensen.disconnect();
        this.connections.delete(sn);
    }
}

function sleep(ms: number) {
    return new Promise(resolve => setTimeout(resolve, ms));
}


export const mgr = new ConnectionManager(Logger);