
/**
 * 模拟 Recorder.getRecordAnalyseData 的功能
 * @param {Int16Array} pcmData PCM 原始数据，s16le 格式
 * @returns {Uint8Array} 长度固定 1024，范围 0–255 的波形数据
 */
function getRecordAnalyseData(pcmData) {
    const BUF_LEN = 800;
    const out = new Uint8Array(BUF_LEN);

    const pcm = [];
    for (let i = 0; i < pcmData.length; i+=2) {
      let h = pcmData[i] & 0xff;
      let l = pcmData[i + 1] & 0xff;
      let p = (l << 8) | h;
      // 如果p是负数，则取反
      if (p & 0x8000) p = (p & 0x7fff) - 0x8000;
      pcm.push(p);
    }
    const len = pcm.length;
    if (len === 0) return out.fill(128); // 空数据，填充中值

    const step = len / BUF_LEN;

    let rst = [];
    for (let i = 0; i < BUF_LEN; i++) {
        const start = Math.floor(i * step);
        const end = Math.min(Math.floor((i + 1) * step), len);

        // 计算这一段的平均值（保持正负号，显示真实波形）
        let sum = 0;
        for (let j = start; j < end; j++)
        {
          sum += Math.abs(pcm[j]);
        }
        const avg = Math.floor(sum / (end - start));
        rst.push(avg);

        // 将平均值从 [-32768, 32767] 映射到 [0, 255]
        // 128 是中心线（静音），0是最负值，255是最正值
        const normalized = avg / 32767; // 归一化到 [0, 1]
        out[i] = Math.max(0, Math.min(127, Math.round(normalized * 127)));
    }
    return out;
}

self.addEventListener('message', async function(e) {
    try {
        const task = e.data;
        
        // pcmdata的值为浮点类型的0~1，需要转换为s16le格式
        const pcmData = new Int16Array(task.pcmdata.length);
        for (let i = 0; i < task.pcmdata.length; i++)
        {
            pcmData[i] = Math.round(task.pcmdata[i] * 32767);
        }
        
        // 生成波形分析数据
        const waveformData = getRecordAnalyseData(pcmData);
        
        // 返回结果给主线程
        self.postMessage({
            id: task.id,
            success: true,
            waveform: waveformData,
        });
    } catch (error) {
        console.error('Worker error:', error);
        
        // 返回错误信息给主线程
        self.postMessage({
            id: task.id,
            success: false,
            error: error.message
        });
    }
}); 