import { useEffect, useState, useRef } from 'react';
import Jensen, { BluetoothDevice } from '..';
import './index.css';
import { Logger } from './Logger'
import { mgr } from './utils/mgr';

const firmwareVersions = [
  { model: 'hidock-p1:mini', version: '2.1.6', url: 'https://jensen.test.hidock.com/firmwares/p1-mini-2.1.6.bin', remark: 'for samsung' },
  { model: 'hidock-p1:mini', version: '2.1.0', url: 'https://jensen.test.hidock.com/firmwares/mini-2.1.0.bin', remark: '' },
  { model: 'hidock-h1', version: '5.2.9', url: 'https://jensen.test.hidock.com/firmwares/hidock-h1-5.2.9.bin', remark: '' },
  { model: 'hidock-h1e', version: '6.2.9', url: 'https://jensen.test.hidock.com/firmwares/hidock-h1e-6.2.9.bin', remark: '' },
  { model: 'hidock-p1', version: '1.3.6', url: 'https://jensen.test.hidock.com/firmwares/p1-1.3.6.bin', remark: '' },
  { model: 'hidock-p1', version: '1.2.25', url: 'https://jensen.test.hidock.com/firmwares/hidock-p1-1.2.25.bin', remark: '' }
];

const sleep = (ms: number) => {
  return new Promise(resolve => setTimeout(resolve, ms));
}

export function Home() {
  const [files, setFiles] = useState<Jensen.FileInfo[]>([]);
  const [devices, setDevices] = useState<Jensen.BluetoothDevice[]>([]);
  const [greeting, setGreeting] = useState<string|null>(null);
  const [firmwares, setFirmwares] = useState<typeof firmwareVersions|null>(null);
  const [logs, setLogs] = useState<string[]>([]);
  const [sn, setSn] = useState<string|null>(null);
  const [liveStates, setLiveStates] = useState<string>('');
  const [rmsLeft, setRmsLeft] = useState<number>(0);
  const [rmsRight, setRmsRight] = useState<number>(0);
  const [liveDelay, setLiveDelay] = useState<number>(100);
  const liveTimeoutRef = useRef<number | null>(null);
  const sendTimeoutRef = useRef<number | null>(null);
  const liveIntervalRef = useRef<number>(liveDelay);
  useEffect(() => { liveIntervalRef.current = liveDelay; }, [liveDelay]);

  mgr.onconnectionstatechanged((state, dinfo) => {
    console.log('onconnectionstatechanged', state, dinfo);
    // alert(JSON.stringify(dinfo));
  });
  mgr.onautoconnect((dinfo) => {
    // console.log('onautoconnect', dinfo);
    setSn(dinfo.sn);
    // alert(JSON.stringify(dinfo));
  });

  const getJensen = () => {
    if (sn == null) return alert('Please connect to a device first'), null;
    return mgr.getInstance(sn);
  }

  useEffect(() => {
    // console.log(mgr);
    mgr.tryconnect();
  }, []);

  useEffect(() => {
    /*
    jensen.connect();
    jensen.onconnect = () => {
      console.log('connect successfully');
      jensen.addWakeupListener(function() {
        console.log('Wake up from sleep?');
      });
      jensen.getDeviceInfo().then((info) => {
        console.log('getDeviceInfo', info);
        alert(info.sn + ' connected');
      }).catch((e) => {
        console.log('getDeviceInfo error', e);
      });
    };
    */
    // jensen.onerror = (e) => {
    //   console.log('onerror', e);
    //   alert('此设备已经在其它已打开的HiNotes网页上建立连接');
    // };

    // 创建日志更新定时器
    const logUpdateInterval = setInterval(() => {
      let last = Logger.messages.slice(-500).reverse();
      let logStr: string[] = last.map((item) => {
        return `[${item.level === 'error' ? 'x' : '*'}] [${new Date(item.time).toLocaleString()}] (${item.module} - ${item.procedure}) ${item.message}`;
      });
      setLogs(logStr);
    }, 1000);

    // 清理函数
    return () => {
      clearInterval(logUpdateInterval);
    };
  }, []);

  useEffect(() => {
    return () => {
      if (liveTimeoutRef.current !== null) {
        window.clearTimeout(liveTimeoutRef.current);
      }
    };
  }, []);

  const info = async () => {
    // alert(sn);
    let jensen = getJensen();
    if (jensen == null) return;
    // console.log('jensen -> ', jensen);
    let info = await jensen?.getDeviceInfo();
    // alert(JSON.stringify(info));
    Logger.info('jensen', 'info', 'Device Info: ' + JSON.stringify(info));
  }

  let bluetoothDevices: BluetoothDevice[] = [];
  const bluetoothScan  = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    setDevices([]);
    setGreeting('scan started at: ' + new Date().toLocaleString());
    let count = prompt('Please Input the Scan Count:', '1');
    if (count === undefined || count === null) return;
    let countNum = parseInt(count);
    if (isNaN(countNum) || countNum <= 0) return alert('Please Input the Correct Scan Count');
    let rst = await jensen.startBluetoothScan(countNum, 5);
    if (rst == null || rst.result != 'success') return alert('start bluetooth scan failed');
    // console.log(devices);
    // setDevices(devices);
    setGreeting('scanning...');
  }

  const scanBluetooth = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    const scan = async () => {
      await jensen.startBluetoothScan(10, 5);
      await sleep(5000);
      await jensen.stopBluetoothScan(5);
      let devices = await jensen.getScanResults(5);
      console.log(devices);
      if (devices == null)
      {
        Logger.error('jensen', 'bluetooth', 'no scan results.');
      }
      else setDevices(devices);

      setTimeout(scan, 5000);
    };
    scan();
  }

  const stopBluetoothScan = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.stopBluetoothScan(5);
    if (rst == null || rst.result != 'success') return alert('stop bluetooth scan failed.');
    setGreeting('scan stopped');
  }

  const getScanResults = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let devices = await jensen.getScanResults(5);
    console.log(devices);
    if (devices == null) return alert('get scan results failed.');
    setDevices(devices);
  }

  const connect = async () => {
    // await jensen.connect();
    // alert(jensen.getModel() + ' connected')
    let info = await mgr.connect();
    if (info && info.sn)
    {
      setSn(info.sn);
      Logger.info('jensen', 'connect', info.sn + ' connected');
    }
    else
    {
      alert('connect failed');
    }
  }

  const disconnectBTDevice = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let r = await jensen.disconnectBTDevice();
    console.log(r);
  }

  const clsBtnConnect: React.CSSProperties = {
    height: '30px',
    padding: '0px 20px',
    position: 'absolute',
    right: '0px',
    top: '3px',
    cursor: 'pointer'
  }

  const clsBleDevice: React.CSSProperties = {
    height: '40px',
    lineHeight: '40px', 
    minWidth: '100%',
    position: 'relative',
    display: 'inline-block'
  }

  const clsBleName: React.CSSProperties = {
    fontSize: '16px',
    fontFamily: 'consolas'
  }

  const doConnectBluetooth = async (mac:string) => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.connectBTDevice(mac, 10);
    alert('connect: ' + rst.result);
  }

  const getTime = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let time = await jensen.getTime();
    // alert(JSON.stringify(time));
    Logger.info('jensen', 'getTime', 'Time: ' + JSON.stringify(time));
  }

  const listFiles = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let fc = await jensen.getFileCount();
    // alert(fc?.count);
    let files = await jensen.listFiles();
    console.log(files);
    setFiles(files)
  }

  const getBluetoothStatus = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let info = await jensen.getBluetoothStatus();
    // alert(JSON.stringify(info));
    Logger.info('jensen', 'getBluetoothStatus', 'Bluetooth: ' + JSON.stringify(info));
  }

  /*
  const readFilePartial = async() => {
    if (files.length == 0) return alert('You don\'t have any recording files, or you haven\'t queried the file list');
    let s0 = prompt('Please Input the File Index (Start from 0):', '0');
    let s1 = prompt('Please Input the Start Position (Start from 0):', '0');
    let s2 = prompt('Please Input the Read Byte Count', '32');
    if (s0 && s1 && s2) console.log();
    else return;
    let index = parseInt(s0);
    let offset = parseInt(s1);
    let length = parseInt(s2);
    if (isNaN(index) && isNaN(offset) || isNaN(length)) return;
    if (index >= files.length) return alert('Please Input the Correct File Index');
    let file = files[index];
    if (offset < 0 || offset > file.length) return alert('Please Input the Correct Read Start Position');
    if (length <= 0) return alert('Please Input the Correct Read Byte Count');
    let data = await jensen.readFile(file.name, offset, length);
    console.log(data);
  }
  */

  const updateDeviceTone = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let resp = await jensen.requestToneUpdate('826d9bac0b535e7babe02b389327a9f2', 1050688);
    if (resp.code != 0x00) return alert(resp.result);
    
    // 下载文件并完成更新处理
    let xhr = new XMLHttpRequest();
    xhr.open('GET', '/sdfs.bin');
    xhr.setRequestHeader('Content-Type', 'application/octet-stream');
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e)
    {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      jensen.updateTone(new Uint8Array(this.response), 30).then((info) => {
        alert(JSON.stringify(info));
      });
    }
    xhr.send();
  }

  const updateUAC = async () => {
    let jensen = getJensen();
    if (jensen == null) return;

    const uac_firmwares = {
      'hidock-h1' : { url : '/firmwares/jensen-1-uac.bin', md5 : '92e66fd8cfd36f09c83fc61491899307' },
      'hidock-h1e' : { url : '/firmwares/jensen-3-uac.bin', md5 : 'c355c5bf8cc8a8da8bea6b6315ad7649' }
    };
    let uac = uac_firmwares[jensen.getModel() as keyof typeof uac_firmwares];
    if (!uac) return alert('No UAC firmware for your device');
    let resp = await jensen.requestUACUpdate(uac.md5, 1024);
    if (resp.code != 0x00) return alert(resp.result);
    
    // 下载文件并完成更新处理
    let xhr = new XMLHttpRequest();
    xhr.open('GET', uac.url);
    xhr.setRequestHeader('Content-Type', 'application/octet-stream');
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e)
    {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      jensen.updateUAC(new Uint8Array(this.response), 30).then((info) => {
        alert(JSON.stringify(info));
      });
    }
    xhr.send();
  }

  const batteryStatus = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let status = await jensen.getBatteryStatus(5);
    // alert(JSON.stringify(status));
    Logger.info('jensen', 'batteryStatus', 'Battery: ' + JSON.stringify(status));
  }

  let filename = '';
  let filelength = 0;
  const transferFile = async () => {
    if (files == null || files.length == 0) return alert('Please Click "List Files" first');
    let idx = prompt('Please Input the File Index (Start from 0):', '0');
    if (idx === undefined || idx === null) return;
    let num = parseInt(idx);
    if (isNaN(num) || num < 0 || num >= files.length) return alert('Please Input the Correct Index');
    let file = files[num];
    // 2025Jun24-165914-Wip00.hda, 232704 @ 19392
    filename = file.name;
    filelength = file.length;

    get_file();
  }

  const get_file = () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let recvBytes = 0;
    let stime = new Date().getTime();
    jensen.getFile(filename, filelength, function(data) {
      // console.log('data', data.length);
      recvBytes += data.length;
      if (recvBytes >= filelength) {
        let cost = new Date().getTime() - stime;
        alert('File Transfer Complete, Cost: ' + cost + ' ms');
      }
    }, function(recvBytes) {
      let percent = Math.floor(recvBytes / filelength * 100);
      // console.log('progress', percent);
    });
  }

  /*
  const test = async () => {
    await jensen.reconnect();
    let rst = await jensen.getDeviceInfo();
    if (rst) alert(rst.sn + ' reconnected...');
    else alert('Something went wrong...');
  }
  */

  const setWebUSBTimeout = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let ts = prompt('Please Input the Timeout (ms):', '10000');
    if (ts === undefined || ts === null) return;
    let timeout = parseInt(ts);
    if (isNaN(timeout) || timeout <= 0) return alert('Please Input the Correct Timeout');
    await jensen.setWebUSBTimeout(timeout, 30);
    Logger.info('jensen', 'setWebUSBTimeout', 'Set Timeout Success');
  }

  const getWebUSBTimeout = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.getWebUSBTimeout(5);
    // alert(JSON.stringify(rst));
    Logger.info('jensen', 'getWebUSBTimeout', 'WebUSB Timeout: ' + JSON.stringify(rst));
  }

  const listFirmwares = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let model = jensen.getModel();
    let versions = firmwareVersions.filter((item) => item.model == model);
    setFirmwares(versions);
  }

  const updateFirmware = async (version: string, url: string) => {
    let jensen = getJensen();
    if (jensen == null) return;
    // let rst = await jensen.requestFirmwareUpgrade(version, url);
    // alert(JSON.stringify(rst));
    if (!confirm('Ready to update firmware, DO NOT close/refresh this page, and wait for the update to complete.\n\n Confirm to continue')) return;
    // 版本号类似于1.2.3，需要转换为0x010203
    // 将类似于"1.2.3"的字符串转换为0x010203的数值
    let versionNumber = 0;
    if (version && typeof version === 'string') {
      let parts = version.split('.');
      if (parts.length === 3) {
        versionNumber = (parseInt(parts[0]) << 16) | (parseInt(parts[1]) << 8) | (parseInt(parts[2]));
      } else {
        alert('invalid version number');
        return;
      }
    } else {
      alert('invalid version number');
      return;
    }

    Logger.info('jensen', 'ota', 'firmware download start');
    let xhr = new XMLHttpRequest();
    xhr.open('GET', url);
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e) {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      Logger.info('jensen', 'ota', 'firmware download complete');
      Logger.info('jensen', 'ota', 'firmware upgrade start');
      jensen.requestFirmwareUpgrade(versionNumber, this.response.byteLength, 30).then((info) => {
        Logger.info('jensen', 'ota', 'firmware upgrade: ' + JSON.stringify(info));
        if (info.result == 'accepted') {
          // alert('firmware upgrade accepted');
          Logger.info('jensen', 'ota', 'firmware transfer start');
          jensen.uploadFirmware(Array.from(new Uint8Array(this.response)), 30).then((info) => {
            if (info == null)
            {
              Logger.info('jensen', 'ota', 'firmware transfer timedout');
            }
            else
            {
              Logger.info('jensen', 'ota', 'firmware transfer complete: ' + JSON.stringify(info));
              Logger.info('jensen', 'ota', 'please wait for the firmware upgrade to complete');
            }
          });
        }
        else
        {
          alert('cannot upgrade firmware: ' + info.result);
        }
      });
    }
    xhr.send();
  }

  const turnPopupOn = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    // await jensen.();
    let rst = await jensen.setNotification(true, 5);
    Logger.info('jensen', 'popup', 'Turn Popup On: ' + JSON.stringify(rst));
  }
  
  const turnPopupOff = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    // await jensen.turnPopupOff();
    let rst = await jensen.setNotification(false, 5);
    Logger.info('jensen', 'popup', 'Turn Popup On: ' + JSON.stringify(rst));
  }

  const getSettings = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.getSettings(5);
    Logger.info('jensen', 'settings', 'Get Settings: ' + JSON.stringify(rst));
  }

  const getPairedDevices = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let devices = await jensen.getPairedDevices(5);
    // alert(JSON.stringify(devices));
    Logger.info('jensen', 'bluetooth', 'Paired Devices: ' + JSON.stringify(devices));
  }

  const clearPairedDevices = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.clearPairedDevices(5);
    Logger.info('jensen', 'bluetooth', 'Clear Paired Devices: ' + JSON.stringify(rst));
  }

  /**
   * 计算双声道PCM数据的RMS和单声道PCM数据
   * @param u8 - 双声道PCM数据
   * @returns - 左声道RMS，右声道RMS，单声道PCM数据
   */
  const rms = (u8: Uint8Array): [number, number, Uint8Array] => {
    if (!u8 || u8.byteLength <= 8) return [0, 0, new Uint8Array(0)];
    // 每帧: L(2B)+R(2B)，总帧数（48k立体声）
    const totalFrames = Math.floor((u8.length - 8) / 4);
    if (totalFrames <= 0) return [0, 0, new Uint8Array(0)];
    // 48k -> 16k，降采样因子为3
    const downFactor = 3;
    const outFrames = Math.floor((totalFrames + downFactor - 1) / downFactor) * 2; // 向上取整
    let sumL = 0;
    let sumR = 0;
    const mono8 = new Uint8Array(outFrames + 8);
    let j = 0, k = 0;
    for (let i = 8; i + 3 < u8.length; i += 4, j++)
    {
      const a = u8[i + 0] & 0xff;
      const b = u8[i + 1] & 0xff;
      const c = u8[i + 2] & 0xff;
      const d = u8[i + 3] & 0xff;
      // 小端 16-bit
      let l = (b << 8) | a;
      let r = (d << 8) | c;
      // 还原为有符号 16-bit
      if (l & 0x8000) l = l - 0x10000;
      if (r & 0x8000) r = r - 0x10000;
      const nl = l / 32768;
      const nr = r / 32768;
      sumL += nl * nl;
      sumR += nr * nr;
      // 双声道合成为单声道（16-bit），再压缩为 Int8
      // C = A + B - (A * B / 32767)
      if (j % downFactor == 0)
      {
        let m = l + r - Math.floor((l * r) / 32767);
        if (m > 32767) m = 32767;
        else if (m < -32768) m = -32768;
        let H = (m >> 8) & 0xff;
        let L = m & 0xff;
        mono8[8 + k++] = (L << 8);
        mono8[8 + k++] = H;
      }
    }
    const leftRms = Math.sqrt(sumL / totalFrames);
    const rightRms = Math.sqrt(sumR / totalFrames);
    // 返回实际长度的视图，避免尾部未填充元素
    return [leftRms, rightRms, mono8];
  }

  const blocksRef = useRef<Uint8Array[]>([]);
  const scheduleLiveTick = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let live = await jensen.getRealtime(1, false, 1);
    // 若无数据直接调度下一次
    if (live?.data == null) return;
    setLiveStates('live: ' + live.data.length + ' rest: ' + live.rest + ' muted: ' + live.muted);
    // 仅计算，不保留引用；如需录音导出再使用 blocksRef.current.push(live.data.slice(8))
    // blocksRef.current.push(live.data.slice(8));
    const [rms1, rms2, mono16k] = rms(live.data);
    // 基于切片内容复制，避免使用 .buffer 造成长度/偏移误用
    blocksRef.current.push(new Uint8Array(mono16k));
    // console.log('live, rms1', rms1, 'rms2', rms2);
    setRmsLeft(Math.floor(rms1 * 4 * 400));
    setRmsRight(Math.floor(rms2 * 4 * 400));

    console.log('live', 'rest: ' + live?.rest);

    // 根据live.rest的值决定timeout的间隔时长，如果>1则为50，否则为100
    let timeout = live.rest > 1 ? 50 : 100;
    liveTimeoutRef.current = window.setTimeout(scheduleLiveTick, Math.max(0, timeout | 0));
  }

  const startSession = async () => {
    
    const jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.startRealtime(2, 1);
    Logger.info('jensen', 'live', 'Start Live: ' + JSON.stringify(rst));

    if (liveTimeoutRef.current !== null) {
      window.clearTimeout(liveTimeoutRef.current);
    }
    liveTimeoutRef.current = window.setTimeout(scheduleLiveTick, 100);

    // 开启发送音频定时器
    if (sendTimeoutRef.current !== null) {  
      window.clearTimeout(sendTimeoutRef.current);
    }
    sendTimeoutRef.current = window.setTimeout(sendLiveAudio, 100);
  }

  function stringToUuidBytes(str: string): Uint8Array {
    const encoder = new TextEncoder();
    const data = encoder.encode(str);
    const hash = new Uint8Array(16);
    // 如果数据长度小于16，用0填充；如果大于16，截断
    for (let i = 0; i < 16; i++) {
      hash[i] = i < data.length ? data[i] : 0;
    }
    return hash;
  }

  const sendLiveAudio = async () => {
    if (blocksRef.current.length == 0) return sendTimeoutRef.current = window.setTimeout(sendLiveAudio, 10);
    console.log('live', 'send audio, rest: ', blocksRef.current.length);
    const audio = blocksRef.current.shift();
    if (audio == null) return console.log('live', 'skip cuz audio is null');
    if (audio.length == 0) return console.log('live', 'skip cuz audio is empty');
    let time = new Date().getTime();
    const timeBytes = u64ToBytesLE(time);
    audio.set(timeBytes, 0);
    wsRef.current?.send(audio.buffer);
    sendTimeoutRef.current = window.setTimeout(sendLiveAudio, 10);
  }

  function u64ToBytesLE(n: number): Uint8Array {
    // 确保是 BigInt
    let bn = BigInt(n);
    const mask = BigInt(0xff);
    const shift = BigInt(8);
    const bytes = new Uint8Array(8);
    for (let i = 0; i < 8; i++) {
      bytes[i] = Number(bn & mask);
      bn = bn >> shift;
    }
    for (let i = 0; i < 4; i++) {
      let tmp = bytes[i];
      bytes[i] = bytes[7 - i];
      bytes[7 - i] = tmp;
    }
    return bytes;
  }

  // 创建一个websocket实例
  const wsRef = useRef<WebSocket | null>(null);
  const startLiveSession = async () => {
    
    const token = 'kL5NmbPEGOptIGbt8jtR3pjXVHT9p774n7N4oQAARb0ctUMnq4bABgBPZ7QjYSUi';
    let url = 'wss://live.test.hidock.com/ws/realtime?accesstoken=' + token + '&mode=room&language=cmn&';
    //         wss://live.test.hidock.com/ws/realtime?accesstoken=&mode=room&language=cmn
    wsRef.current = new WebSocket(url);
    wsRef.current.onopen = () => {
      console.log('websocket connected');
      const config = {
        "type": "config",
        "sequence": String(new Date().getTime()),
        "body": {
            "language": "cmn",
            "enable_speaker_diarization": true,
            "translation": {
              "type": "one_way",
              "language": "cmn",
              "target_language": "eng"
            },
            "language_hints": ["cmn"],
            "create_note": false
        }
      };
      wsRef.current?.send(JSON.stringify(config));
    }
    wsRef.current.onmessage = (event) => {
      // console.log('websocket message', event.data);
      // {"code":0,"message":"","sequence":"1764835254003","data":{"type":"config","type":"config","session_id":"91e0c9710467faaa57c21eb8b8c73a61","status":"created"}}
      let msg = JSON.parse(event.data);
      if (msg.code != 0) return console.error('websocket message error', msg);
      let data = msg.data;
      if (data.type == 'config' && data.status == 'created')
      {
        console.log('websocket session created', data);
        startSession();
      }
      if (data.type == 'partial')
      {
        console.log('live', data.text);
      }
      if (data.type == 'final')
      {
        console.log('live', data.text);
      }
    }
    wsRef.current.onerror = (event) => {
      console.log('websocket error', event);
    }
    wsRef.current.onclose = () => {
      console.log('websocket closed');
    }
  }
  
  const startLive = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    
    startLiveSession();
  }
  
  const stopLive = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.stopRealtime(1);
    Logger.info('jensen', 'live', 'Stop Live: ' + JSON.stringify(rst));
    if (liveTimeoutRef.current !== null) {
      window.clearTimeout(liveTimeoutRef.current);
      liveTimeoutRef.current = null;
    }
    console.log('live stopped');
    /*
    const totalSamples = blocksRef.current.reduce((sum, arr) => sum + arr.length, 0);
    const monoAll = new Uint8Array(totalSamples);
    let offsetMono = 0;
    for (const arr of blocksRef.current) {
      monoAll.set(arr, offsetMono);
      offsetMono += arr.length;
    }
    let blob = new Blob([monoAll.buffer], { type: 'audio/pcm' });
    let url = URL.createObjectURL(blob);
    let a = document.createElement('a');
    a.href = url;
    a.download = 'live.pcm';
    a.click();
    URL.revokeObjectURL(url);
    */
  }

  return (
    <>
    <div className="btn-container" style={{ display: 'flex', flexDirection: 'row', gap: '16px', padding: '16px', alignItems: 'center', flexWrap: 'wrap' }}>
      <button onClick={connect}>Connect</button>
      <button onClick={info}>Info</button>
      <button onClick={getTime}>Get Time</button>
      <button onClick={listFiles}>List Files</button>
      <button onClick={transferFile}>Transfer</button>
      <button onClick={batteryStatus}>Battery</button>
      <button onClick={getBluetoothStatus}>Bluetooth Status</button>
      <button onClick={bluetoothScan}>Start Scan</button>
      <button onClick={stopBluetoothScan}>Stop Scan</button>
      <button onClick={getScanResults}>Scan Results</button>
      <button onClick={scanBluetooth}>Scan</button>
      <button onClick={getPairedDevices}>Paired Devices</button>
      <button onClick={clearPairedDevices}>Clear Paired</button>
      {
      /*
      <button onClick={reconnectDevice}>Connect Device</button>
      */
      }
      <button onClick={disconnectBTDevice}>Disconnect</button>
      <button onClick={updateDeviceTone}>Update Tone</button>
      <button onClick={updateUAC}>Update UAC</button>
      <button onClick={setWebUSBTimeout}>Set Timeout</button>
      <button onClick={getWebUSBTimeout}>Get Timeout</button>
      <button onClick={listFirmwares}>Firmwares</button>
      <button onClick={turnPopupOn}>Turn Popup On</button>
      <button onClick={turnPopupOff}>Turn Popup Off</button>
      <button onClick={getSettings}>Get Settings</button>
      <button onClick={startLive}>Start Live</button>
      <button onClick={stopLive}>Stop Live</button>
      <span style={{ marginLeft: '8px' }}>Interval(ms): </span>
      <input
        type="number"
        value={liveDelay}
        onChange={(e) => {
          const v = parseInt(e.target.value || '0');
          setLiveDelay(isNaN(v) ? 0 : v);
        }}
        style={{ width: '90px' }}
      />
    </div>
    <div className="result-container">
      <div className="list-container">
        {
          firmwares && firmwares.length ? (
            <div style={{ padding: '0px 0px 0px 30px', marginBottom: '20px' }}>
              <h3>Firmwares: </h3>
              <ol style={{ padding: '0px 0px 0px 20px', 'listStyle': 'none', lineHeight: '20px' }}>
                {
                  firmwares.map((item, index) => {
                    return <li key={item.version}>{item.version} {item.remark ? '- (' + item.remark + ')' : ''} - <a href="#" onClick={() => { updateFirmware(item.version, item.url) }}>Update</a></li>
                  })
                }
              </ol>
            </div>
          ) : (<></>)
        }
        {
          liveStates ? (<>
          <div>
              <h3>{liveStates}</h3>
              <div className="bar"><div style={{ width: rmsLeft + 'px' }}></div></div>
              <div className="bar"><div style={{ width: rmsRight + 'px' }}></div></div>
          </div>
          </>) : <></>
        }
        <div id="files" style={{ padding: '0px 0px 0px 30px', marginBottom: '20px' }}>
          <h3>Files: </h3>
          <ol style={{ padding: '0px 0px 0px 30px', 'listStyle': 'none' }}>
            { files.map((item, index) => {
              return <li key={item.name}>{index} - {item?.name}, {item?.length} @ {item.duration}</li>
            })}
          </ol>
        </div>
        <div style={{ padding: '0px 0px 0px 30px', width: '800px' }}>
          <h3>Bluetooth Device List: </h3>
          {
            greeting ? <div>{greeting}</div> : <></>
          }
          {
            devices && devices.length ? (
              devices.map((item, index) => {
                return (
                  <div style={clsBleDevice}><span style={clsBleName}>({item.mac}) {item.name}</span><button style={clsBtnConnect} onClick={() => { doConnectBluetooth(item.mac) }}>Connect</button></div>
                );
              })
            ) : (
              <div>no devices...</div>
            )
          }
        </div>
      </div>
      <div className="log-container">
        <textarea id="log" style={{ width: '100%', height: '100%' }} value={logs.join('\n')} readOnly />
      </div>
    </div>
    </>
  );
}
