import { useEffect, useState } from 'react';
import Jensen, { BluetoothDevice } from '..';
import './index.css';
import { Logger } from './Logger'
import { mgr } from './utils/mgr';

const firmwareVersions = [
  { model: 'hidock-p1:mini', version: '2.1.6', url: 'https://jensen.test.hidock.com/firmwares/p1-mini-2.1.6.bin', remark: 'for samsung' },
  { model: 'hidock-p1:mini', version: '2.1.0', url: 'https://jensen.test.hidock.com/firmwares/mini-2.1.0.bin', remark: '' },
  { model: 'hidock-h1', version: '5.2.9', url: 'https://jensen.test.hidock.com/firmwares/hidock-h1-5.2.9.bin', remark: '' },
  { model: 'hidock-h1e', version: '6.2.9', url: 'https://jensen.test.hidock.com/firmwares/hidock-h1e-6.2.9.bin', remark: '' },
  { model: 'hidock-p1', version: '1.3.6', url: 'https://jensen.test.hidock.com/firmwares/p1-1.3.6.bin', remark: '' },
  { model: 'hidock-p1', version: '1.2.25', url: 'https://jensen.test.hidock.com/firmwares/hidock-p1-1.2.25.bin', remark: '' }
];


export function Home() {
  const [files, setFiles] = useState<Jensen.FileInfo[]>([]);
  const [devices, setDevices] = useState<Jensen.BluetoothDevice[]>([]);
  const [greeting, setGreeting] = useState<string|null>(null);
  const [firmwares, setFirmwares] = useState<typeof firmwareVersions|null>(null);
  const [logs, setLogs] = useState<string[]>([]);
  const [sn, setSn] = useState<string|null>(null);

  mgr.onconnectionstatechanged((state, dinfo) => {
    console.log('onconnectionstatechanged', state, dinfo);
    // alert(JSON.stringify(dinfo));
  });
  mgr.onautoconnect((dinfo) => {
    // console.log('onautoconnect', dinfo);
    setSn(dinfo.sn);
    // alert(JSON.stringify(dinfo));
  });

  const getJensen = () => {
    if (sn == null) return alert('Please connect to a device first'), null;
    return mgr.getInstance(sn);
  }

  useEffect(() => {
    // console.log(mgr);
    mgr.tryconnect();
  }, []);

  useEffect(() => {
    /*
    jensen.connect();
    jensen.onconnect = () => {
      console.log('connect successfully');
      jensen.addWakeupListener(function() {
        console.log('Wake up from sleep?');
      });
      jensen.getDeviceInfo().then((info) => {
        console.log('getDeviceInfo', info);
        alert(info.sn + ' connected');
      }).catch((e) => {
        console.log('getDeviceInfo error', e);
      });
    };
    */
    // jensen.onerror = (e) => {
    //   console.log('onerror', e);
    //   alert('此设备已经在其它已打开的HiNotes网页上建立连接');
    // };

    // 创建日志更新定时器
    const logUpdateInterval = setInterval(() => {
      // 从Logger.messages中获取最后500条记录
      let last = Logger.messages.slice(-500).reverse();
      // 将last转换为字符串数组
      let logStr: string[] = last.map((item) => {
        return `[${item.level === 'error' ? 'x' : '*'}] [${new Date(item.time).toLocaleString()}] (${item.module} - ${item.procedure}) ${item.message}`;
      });
      setLogs(logStr);
    }, 1000);

    // 清理函数
    return () => {
      clearInterval(logUpdateInterval);
    };
  }, []);

  const info = async () => {
    // alert(sn);
    let jensen = getJensen();
    if (jensen == null) return;
    // console.log('jensen -> ', jensen);
    let info = await jensen?.getDeviceInfo();
    // alert(JSON.stringify(info));
    Logger.info('jensen', 'info', 'Device Info: ' + JSON.stringify(info));
  }

  let bluetoothDevices: BluetoothDevice[] = [];
  const bluetoothScan  = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    setDevices([]);
    setGreeting('scan started at: ' + new Date().toLocaleString());
    let count = prompt('Please Input the Scan Count:', '1');
    if (count === undefined || count === null) return;
    let countNum = parseInt(count);
    if (isNaN(countNum) || countNum <= 0) return alert('Please Input the Correct Scan Count');
    let rst = await jensen.startBluetoothScan(countNum, 5);
    if (rst == null || rst.result != 'success') return alert('start bluetooth scan failed');
    // console.log(devices);
    // setDevices(devices);
    setGreeting('scanning...');
  }

  const stopBluetoothScan = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.stopBluetoothScan(5);
    if (rst == null || rst.result != 'success') return alert('stop bluetooth scan failed.');
    setGreeting('scan stopped');
  }

  const getScanResults = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let devices = await jensen.getScanResults(5);
    if (devices == null) return alert('get scan results failed.');
    setDevices(devices);
  }

  const connect = async () => {
    // await jensen.connect();
    // alert(jensen.getModel() + ' connected')
    let info = await mgr.connect();
    if (info && info.sn)
    {
      setSn(info.sn);
      Logger.info('jensen', 'connect', info.sn + ' connected');
    }
    else
    {
      alert('connect failed');
    }
  }

  const disconnectBTDevice = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let r = await jensen.disconnectBTDevice();
    console.log(r);
  }

  const clsBtnConnect = {
    height: '30px',
    padding: '0px 20px',
    position: 'absolute',
    right: '0px',
    top: '3px',
    cursor: 'pointer'
  }

  const clsBleDevice = {
    height: '40px',
    lineHeight: '40px', 
    minWidth: '500px',
    position: 'relative',
    display: 'inline-block'
  }

  const clsBleName = {
    fontSize: '16px',
    fontFamily: 'consolas'
  }

  const doConnectBluetooth = async (mac:string) => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.connectBTDevice(mac, 10);
    alert('connect: ' + rst.result);
  }

  const getTime = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let time = await jensen.getTime();
    // alert(JSON.stringify(time));
    Logger.info('jensen', 'getTime', 'Time: ' + JSON.stringify(time));
  }

  const listFiles = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let fc = await jensen.getFileCount();
    // alert(fc?.count);
    let files = await jensen.listFiles();
    console.log(files);
    setFiles(files)
  }

  const getBluetoothStatus = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let info = await jensen.getBluetoothStatus();
    // alert(JSON.stringify(info));
    Logger.info('jensen', 'getBluetoothStatus', 'Bluetooth: ' + JSON.stringify(info));
  }

  /*
  const readFilePartial = async() => {
    if (files.length == 0) return alert('You don\'t have any recording files, or you haven\'t queried the file list');
    let s0 = prompt('Please Input the File Index (Start from 0):', '0');
    let s1 = prompt('Please Input the Start Position (Start from 0):', '0');
    let s2 = prompt('Please Input the Read Byte Count', '32');
    if (s0 && s1 && s2) console.log();
    else return;
    let index = parseInt(s0);
    let offset = parseInt(s1);
    let length = parseInt(s2);
    if (isNaN(index) && isNaN(offset) || isNaN(length)) return;
    if (index >= files.length) return alert('Please Input the Correct File Index');
    let file = files[index];
    if (offset < 0 || offset > file.length) return alert('Please Input the Correct Read Start Position');
    if (length <= 0) return alert('Please Input the Correct Read Byte Count');
    let data = await jensen.readFile(file.name, offset, length);
    console.log(data);
  }
  */

  const updateDeviceTone = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let resp = await jensen.requestToneUpdate('826d9bac0b535e7babe02b389327a9f2', 1050688);
    if (resp.code != 0x00) return alert(resp.result);
    
    // 下载文件并完成更新处理
    let xhr = new XMLHttpRequest();
    xhr.open('GET', '/sdfs.bin');
    xhr.setRequestHeader('Content-Type', 'application/octet-stream');
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e)
    {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      jensen.updateTone(new Uint8Array(this.response), 30).then((info) => {
        alert(JSON.stringify(info));
      });
    }
    xhr.send();
  }

  const updateUAC = async () => {
    let jensen = getJensen();
    if (jensen == null) return;

    const uac_firmwares = {
      'hidock-h1' : { url : '/firmwares/jensen-1-uac.bin', md5 : '92e66fd8cfd36f09c83fc61491899307' },
      'hidock-h1e' : { url : '/firmwares/jensen-3-uac.bin', md5 : 'c355c5bf8cc8a8da8bea6b6315ad7649' }
    };
    let uac = uac_firmwares[jensen.getModel() as keyof typeof uac_firmwares];
    if (!uac) return alert('No UAC firmware for your device');
    let resp = await jensen.requestUACUpdate(uac.md5, 1024);
    if (resp.code != 0x00) return alert(resp.result);
    
    // 下载文件并完成更新处理
    let xhr = new XMLHttpRequest();
    xhr.open('GET', uac.url);
    xhr.setRequestHeader('Content-Type', 'application/octet-stream');
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e)
    {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      jensen.updateUAC(new Uint8Array(this.response), 30).then((info) => {
        alert(JSON.stringify(info));
      });
    }
    xhr.send();
  }

  const batteryStatus = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let status = await jensen.getBatteryStatus(5);
    // alert(JSON.stringify(status));
    Logger.info('jensen', 'batteryStatus', 'Battery: ' + JSON.stringify(status));
  }

  let filename = '';
  let filelength = 0;
  const transferFile = async () => {
    if (files == null || files.length == 0) return alert('Please Click "List Files" first');
    let idx = prompt('Please Input the File Index (Start from 0):', '0');
    if (idx === undefined || idx === null) return;
    let num = parseInt(idx);
    if (isNaN(num) || num < 0 || num >= files.length) return alert('Please Input the Correct Index');
    let file = files[num];
    // 2025Jun24-165914-Wip00.hda, 232704 @ 19392
    filename = file.name;
    filelength = file.length;

    get_file();
  }

  const get_file = () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let recvBytes = 0;
    let stime = new Date().getTime();
    jensen.getFile(filename, filelength, function(data) {
      // console.log('data', data.length);
      recvBytes += data.length;
      if (recvBytes >= filelength) {
        let cost = new Date().getTime() - stime;
        alert('File Transfer Complete, Cost: ' + cost + ' ms');
      }
    }, function(recvBytes) {
      let percent = Math.floor(recvBytes / filelength * 100);
      // console.log('progress', percent);
    });
  }

  /*
  const test = async () => {
    await jensen.reconnect();
    let rst = await jensen.getDeviceInfo();
    if (rst) alert(rst.sn + ' reconnected...');
    else alert('Something went wrong...');
  }
  */

  const setWebUSBTimeout = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let ts = prompt('Please Input the Timeout (ms):', '10000');
    if (ts === undefined || ts === null) return;
    let timeout = parseInt(ts);
    if (isNaN(timeout) || timeout <= 0) return alert('Please Input the Correct Timeout');
    await jensen.setWebUSBTimeout(timeout, 30);
    Logger.info('jensen', 'setWebUSBTimeout', 'Set Timeout Success');
  }

  const getWebUSBTimeout = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.getWebUSBTimeout(5);
    // alert(JSON.stringify(rst));
    Logger.info('jensen', 'getWebUSBTimeout', 'WebUSB Timeout: ' + JSON.stringify(rst));
  }

  const listFirmwares = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let model = jensen.getModel();
    let versions = firmwareVersions.filter((item) => item.model == model);
    setFirmwares(versions);
  }

  const updateFirmware = async (version: string, url: string) => {
    let jensen = getJensen();
    if (jensen == null) return;
    // let rst = await jensen.requestFirmwareUpgrade(version, url);
    // alert(JSON.stringify(rst));
    if (!confirm('Ready to update firmware, DO NOT close/refresh this page, and wait for the update to complete.\n\n Confirm to continue')) return;
    // 版本号类似于1.2.3，需要转换为0x010203
    // 将类似于"1.2.3"的字符串转换为0x010203的数值
    let versionNumber = 0;
    if (version && typeof version === 'string') {
      let parts = version.split('.');
      if (parts.length === 3) {
        versionNumber = (parseInt(parts[0]) << 16) | (parseInt(parts[1]) << 8) | (parseInt(parts[2]));
      } else {
        alert('invalid version number');
        return;
      }
    } else {
      alert('invalid version number');
      return;
    }

    Logger.info('jensen', 'ota', 'firmware download start');
    let xhr = new XMLHttpRequest();
    xhr.open('GET', url);
    xhr.responseType = 'arraybuffer';
    xhr.onload = function(e) {
      if (this.status != 200) return alert('Http Error: ' + this.status);
      Logger.info('jensen', 'ota', 'firmware download complete');
      Logger.info('jensen', 'ota', 'firmware upgrade start');
      jensen.requestFirmwareUpgrade(versionNumber, this.response.byteLength, 30).then((info) => {
        Logger.info('jensen', 'ota', 'firmware upgrade: ' + JSON.stringify(info));
        if (info.result == 'accepted') {
          // alert('firmware upgrade accepted');
          Logger.info('jensen', 'ota', 'firmware transfer start');
          jensen.uploadFirmware(Array.from(new Uint8Array(this.response)), 30).then((info) => {
            if (info == null)
            {
              Logger.info('jensen', 'ota', 'firmware transfer timedout');
            }
            else
            {
              Logger.info('jensen', 'ota', 'firmware transfer complete: ' + JSON.stringify(info));
              Logger.info('jensen', 'ota', 'please wait for the firmware upgrade to complete');
            }
          });
        }
        else
        {
          alert('cannot upgrade firmware: ' + info.result);
        }
      });
    }
    xhr.send();
  }

  const turnPopupOn = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    // await jensen.();
    let rst = await jensen.setNotification(true, 5);
    Logger.info('jensen', 'popup', 'Turn Popup On: ' + JSON.stringify(rst));
  }
  
  const turnPopupOff = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    // await jensen.turnPopupOff();
    let rst = await jensen.setNotification(false, 5);
    Logger.info('jensen', 'popup', 'Turn Popup On: ' + JSON.stringify(rst));
  }

  const getSettings = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let rst = await jensen.getSettings(5);
    Logger.info('jensen', 'settings', 'Get Settings: ' + JSON.stringify(rst));
  }

  const getPairedDevices = async () => {
    let jensen = getJensen();
    if (jensen == null) return;
    let devices = await jensen.getPairedDevices(5);
    alert(JSON.stringify(devices));
  }

  return (
    <>
    <div className="btn-container" style={{ display: 'flex', flexDirection: 'row', gap: '16px', padding: '16px', alignItems: 'center', flexWrap: 'wrap' }}>
      <button onClick={connect}>Connect</button>
      <button onClick={info}>Info</button>
      <button onClick={getTime}>Get Time</button>
      <button onClick={listFiles}>List Files</button>
      <button onClick={transferFile}>Transfer</button>
      <button onClick={batteryStatus}>Battery</button>
      <button onClick={getBluetoothStatus}>Bluetooth Status</button>
      <button onClick={bluetoothScan}>Start Scan</button>
      <button onClick={stopBluetoothScan}>Stop Scan</button>
      <button onClick={getScanResults}>Scan Results</button>
      {
      /*
      <button onClick={getPairedDevices}>Paired Devices</button>
      <button onClick={reconnectDevice}>Connect Device</button>
      */
      }
      <button onClick={disconnectBTDevice}>Disconnect</button>
      <button onClick={updateDeviceTone}>Update Tone</button>
      <button onClick={updateUAC}>Update UAC</button>
      <button onClick={setWebUSBTimeout}>Set Timeout</button>
      <button onClick={getWebUSBTimeout}>Get Timeout</button>
      <button onClick={listFirmwares}>Firmwares</button>
      <button onClick={turnPopupOn}>Turn Popup On</button>
      <button onClick={turnPopupOff}>Turn Popup Off</button>
      <button onClick={getSettings}>Get Settings</button>
    </div>
    <div className="result-container">
      <div className="list-container">
        {
          firmwares && firmwares.length ? (
            <div style={{ padding: '0px 0px 0px 30px', marginBottom: '20px' }}>
              <h3>Firmwares: </h3>
              <ol style={{ padding: '0px 0px 0px 20px', 'listStyle': 'none', lineHeight: '20px' }}>
                {
                  firmwares.map((item, index) => {
                    return <li key={item.version}>{item.version} {item.remark ? '- (' + item.remark + ')' : ''} - <a href="#" onClick={() => { updateFirmware(item.version, item.url) }}>Update</a></li>
                  })
                }
              </ol>
            </div>
          ) : (<></>)
        }
        <div id="files" style={{ padding: '0px 0px 0px 30px', marginBottom: '20px' }}>
          <h3>Files: </h3>
          <ol style={{ padding: '0px 0px 0px 30px', 'listStyle': 'none' }}>
            { files.map((item, index) => {
              return <li key={item.name}>{index} - {item?.name}, {item?.length} @ {item.duration}</li>
            })}
          </ol>
        </div>
        <div style={{ padding: '0px 0px 0px 30px', width: '500px' }}>
          <h3>Bluetooth Device List: </h3>
          {
            greeting ? <div>{greeting}</div> : <></>
          }
          {
            devices && devices.length ? (
              devices.map((item, index) => {
                return (
                  <div style={clsBleDevice}><span style={clsBleName}>({item.mac}) {item.name}</span><button style={clsBtnConnect} onClick={() => { doConnectBluetooth(item.mac) }}>Connect</button></div>
                );
              })
            ) : (
              <div>no devices...</div>
            )
          }
        </div>
      </div>
      <div className="log-container">
        <textarea id="log" style={{ width: '100%', height: '100%' }} value={logs.join('\n')} readOnly />
      </div>
    </div>
    </>
  );
}
