# Jensen 设备管理开发指南

本文档介绍如何使用 Jensen 库进行 HiDock 设备的连接管理、任务调度和状态监控。

## 📋 目录

- [核心概念](#核心概念)
- [快速开始](#快速开始)
- [设备连接管理](#设备连接管理)
- [任务系统](#任务系统)
- [设备状态管理](#设备状态管理)
- [完整示例](#完整示例)
- [API 参考](#api-参考)

## 核心概念

### 架构概览

```
┌─────────────────────────────────────────────────────────────────┐
│                     ConnectionManager (mgr)                      │
├─────────────────────────────────────────────────────────────────┤
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │   Device 1  │  │   Device 2  │  │   Device N  │              │
│  │   (Jensen)  │  │   (Jensen)  │  │   (Jensen)  │              │
│  └──────┬──────┘  └──────┬──────┘  └──────┬──────┘              │
│         │                │                │                      │
│  ┌──────▼──────┐  ┌──────▼──────┐  ┌──────▼──────┐              │
│  │TaskManager 1│  │TaskManager 2│  │TaskManager N│              │
│  ├─────────────┤  ├─────────────┤  ├─────────────┤              │
│  │ StateStore  │  │ StateStore  │  │ StateStore  │              │
│  │ TimerTasks  │  │ TimerTasks  │  │ TimerTasks  │              │
│  │ UserTask    │  │ UserTask    │  │ UserTask    │              │
│  └─────────────┘  └─────────────┘  └─────────────┘              │
└─────────────────────────────────────────────────────────────────┘
```

### 核心类说明

| 类名 | 说明 |
|------|------|
| `ConnectionManager` | 设备连接管理器，负责多设备的连接、断开和实例管理 |
| `TaskManager` | 任务调度器，管理初始化任务、定时任务、后台任务和用户任务 |
| `DeviceStateStore` | 设备状态存储，提供状态读写和订阅功能 |
| `Jensen` | 设备通信实例，提供所有与设备交互的 API |

### 任务类型

| 任务类型 | 说明 | 典型用途 |
|----------|------|----------|
| `InitializeTask` | 初始化任务，设备连接后执行一次 | 获取设备信息、同步时间 |
| `TimerTask` | 定时任务，按指定间隔循环执行 | 轮询电池状态、文件列表 |
| `BackgroundTask` | 后台任务，带调度逻辑的周期任务 | 录音传输及后续的处理 |
| `UserTask` | 用户任务，**排它性任务**，用于用户交互操作 | 用户点击引发的需要即时完成的操作指令 |

> ⚠️ **重要**：`UserTask` 是排它性的，当一个 UserTask 在执行时，所有定时任务和后台任务会暂停，以确保用户操作不被干扰。

## 快速开始

### 基本使用

```typescript
import { mgr } from './utils/mgr';

// 1. 监听自动连接事件
mgr.onautoconnect((dinfo) => {
    console.log('设备已连接:', dinfo.sn);
});

// 2. 尝试连接已授权的设备，可以直接触发完成自动连接
await mgr.tryconnect();

// 3. 尝试连接新设备，需要从用户事件触发
await mgr.connect();
```

## 设备连接管理

### 自动连接

当页面加载时，调用 `tryconnect()` 会自动连接所有已授权的 HiDock 设备：

```typescript
import { mgr } from './utils/mgr';

// 注册连接事件监听器
mgr.onautoconnect((dinfo) => {
    console.log('设备序列号:', dinfo.sn);
    console.log('固件版本:', dinfo.versionCode);
});

// 尝试自动连接
await mgr.tryconnect();
```

### 手动连接

如果需要让用户选择设备进行连接：

```typescript
const dinfo = await mgr.connect();
if (dinfo) {
    console.log('手动连接成功:', dinfo.sn);
}
```

### 监听连接状态变化

```typescript
mgr.onconnectionstatechanged((state, dinfo) => {
    switch (state) {
        case 'init':
            console.log('正在初始化设备...');
            break;
        case 'connected':
            console.log('设备已连接:', dinfo);
            break;
        case 'reconnected':
            console.log('设备已重新连接:', dinfo);
            break;
        case 'disconnected':
            console.log('设备已断开:', dinfo);
            break;
        case 'connect-timeout':
            console.log('连接超时');
            break;
    }
});
```

### 断开连接

```typescript
await mgr.close(sn);
```

## 任务系统

### UserTask（用户任务）

UserTask 是用于用户交互操作的**排它性任务**。当你需要执行用户操作（如下载文件、修改设置）时，必须先注册 UserTask。

#### 为什么需要 UserTask？

1. **防止冲突**：确保用户操作不会被定时任务打断
2. **资源独占**：在 UserTask 执行期间，后台任务会暂停
3. **超时保护**：空闲超过 30 秒的 UserTask 会自动释放

#### 注册和释放 UserTask

```typescript
// 注册 UserTask（带超时等待）
const success = await mgr.registerTask(sn, 'my-task-id');
if (!success) {
    console.error('注册任务失败，可能有其他任务正在执行');
    return;
}

try {
    // 执行用户操作...
} finally {
    // 释放 UserTask
    await mgr.unregisterTask(sn, 'my-task-id');
}
```

#### 使用 getInstance 获取设备实例

获取 Jensen 实例时**必须传入 UserTask 标签**，这样所有操作都会自动验证任务有效性，这种方式更适合较复杂的场景：

```typescript
// 正确用法：传入 userTaskTag
const jensen = mgr.getInstance(sn, userTaskTag);
if (jensen == null) return alert('设备未连接');

// 调用方法时会自动验证 UserTask
const time = await jensen.getTime(1);
```

#### 使用 executeWithUserTask（推荐）

更安全的方式是使用 `executeWithUserTask`，它会自动验证任务并执行回调，这种方式更适合较简单的场景：

```typescript
const time = await mgr.executeWithUserTask(sn, userTaskTag, async (jensen) => {
    return await jensen.getTime(1);
});
```

#### 保持任务活跃

如果 UserTask 长时间不活动（超过 30 秒），会被自动释放。对于长时间操作，需要定期调用保活方法：

```typescript
// 保持任务活跃
mgr.keepUserTaskAlive(sn, userTaskTag);
```

### 定时任务（TimerTask）

定时任务会在设备连接后按指定间隔循环执行。内置的定时任务包括：

- `battery-status`：每秒同步电池状态
- `get-device-time`：每秒获取设备时间
- `recording-status`：每秒检查录音状态
- `file-list-sync`：每秒检查文件列表变化

#### 注册自定义定时任务

```typescript
mgr.registerTimerTask(sn, {
    tag: 'my-timer-task',
    interval: 5000,  // 5秒执行一次
    task: async (jensen, store) => {
        const settings = await jensen?.getSettings(1);
        if (settings) {
            store.set('settings', settings);
        }
    }
});
```

### 初始化任务（InitializeTask）

初始化任务在设备首次连接时执行一次。内置的初始化任务包括：

- `time-sync`：同步设备时间
- `get-settings`：获取设备设置
- `get-card-info`：获取存储卡信息
- `battery-status`：获取电池状态

#### 注册自定义初始化任务

```typescript
mgr.registerInitializeTask(sn, {
    tag: 'my-init-task',
    task: async (jensen, store) => {
        const info = await jensen?.getDeviceInfo(1);
        store.set('deviceInfo', info);
    }
});
```

## 设备状态管理

### DeviceStateStore

每个设备都有一个 `DeviceStateStore` 实例，用于存储设备状态。状态变更会自动通知所有订阅者。

### 可用的状态键

```typescript
interface DeviceStateSchema {
    deviceInfo: DeviceInfo | null;
    time: { deviceTime: string; syncTime: Date } | null;
    settings: { autoRecord: boolean; autoPlay: boolean; notification?: boolean } | null;
    cardInfo: { used: number; capacity: number; status: string } | null;
    batteryStatus: BatteryStatus | null;
    bluetoothStatus: BluetoothStatus | null;
    files: FileInfo[] | null;
    recording: FileInfo | null;
    // ... 可动态扩展
}
```

### 订阅状态变更

```typescript
// 订阅电池状态变化
mgr.subscribeDeviceState(sn, 'battery-status', (key, newValue, oldValue) => {
    console.log('电池状态更新:', newValue);
});

// 订阅文件列表变化
mgr.subscribeDeviceState(sn, 'files', (key, newValue, oldValue) => {
    console.log('文件列表更新:', newValue);
});

// 订阅录音状态变化
mgr.subscribeDeviceState(sn, 'recording', (key, newValue, oldValue) => {
    if (newValue && !oldValue) {
        console.log('开始录音:', newValue.name);
    } else if (!newValue && oldValue) {
        console.log('录音结束');
    }
});
```

### 直接读取状态

```typescript
// 获取当前电池状态
const battery = mgr.getDeviceState(sn, 'battery-status');

// 获取文件列表
const files = mgr.getDeviceState(sn, 'files');

// 检查状态是否过期（超过5秒）
const isStale = mgr.isDeviceStateStale(sn, 'battery-status', 5000);
```

### 手动设置状态

```typescript
mgr.setDeviceState(sn, 'settings', {
    autoRecord: true,
    autoPlay: false
});
```

## 完整示例

以下是一个完整的 React 组件示例，展示了所有核心功能的使用：

```typescript
import { mgr } from './utils/mgr';
import { useEffect, useState } from 'react';
import Jensen from '../jensen';

export function DeviceManager() {
    // 状态定义
    const [sn, setSn] = useState<string | null>(null);
    const [time, setTime] = useState<string | null>(null);
    const [battery, setBattery] = useState<string | null>(null);
    const [userTask, setUserTask] = useState<string | null>(null);
    const [files, setFiles] = useState<Jensen.FileInfo[]>([]);
    const [recording, setRecording] = useState<Jensen.FileInfo | null>(null);

    // 设备连接和状态订阅
    useEffect(() => {
        mgr.onautoconnect((dinfo) => {
            setSn(dinfo.sn);
            
            // 订阅时间状态变化
            mgr.subscribeDeviceState(dinfo.sn, 'time', (key, newValue) => {
                setTime(JSON.stringify(newValue));
            });
            
            // 订阅电池状态变化
            mgr.subscribeDeviceState(dinfo.sn, 'battery-status', (key, newValue) => {
                setBattery(JSON.stringify(newValue));
            });
            
            // 订阅文件列表变化
            mgr.subscribeDeviceState(dinfo.sn, 'files', (key, newValue) => {
                setFiles(newValue as Jensen.FileInfo[]);
            });
            
            // 订阅录音状态变化
            mgr.subscribeDeviceState(dinfo.sn, 'recording', (key, newValue) => {
                setRecording(newValue as Jensen.FileInfo | null);
            });
        });
    }, []);

    // 连接设备
    const doConnect = async () => {
        await mgr.tryconnect();
    };

    // 注册 UserTask
    const doRequestUserTask = async () => {
        if (sn == null) return alert('请先连接设备');
        
        const tid = 't-' + new Date().getTime();
        const success = await mgr.registerTask(sn, tid);
        
        if (!success) {
            return alert('注册任务失败');
        }
        
        setUserTask(tid);
        console.log('UserTask 已注册:', tid);
    };

    // 释放 UserTask
    const doReleaseUserTask = async () => {
        if (sn == null) return alert('请先连接设备');
        if (userTask == null) return alert('请先注册 UserTask');
        
        await mgr.unregisterTask(sn, userTask);
        setUserTask(null);
        console.log('UserTask 已释放');
    };

    // 获取设备时间（需要 UserTask）
    const doGetTime = async () => {
        if (sn == null) return alert('请先连接设备');
        if (userTask == null) return alert('请先注册 UserTask');
        
        try {
            // 方式一：使用 getInstance
            const jensen = mgr.getInstance(sn, userTask);
            if (jensen == null) return alert('设备未连接');
            const time = await jensen.getTime(1);
            alert('设备时间: ' + time.time);
            
            // 方式二：使用 executeWithUserTask（推荐）
            // const time = await mgr.executeWithUserTask(sn, userTask, async (jensen) => {
            //     return await jensen.getTime(1);
            // });
            // alert('设备时间: ' + time.time);
        } catch (err: any) {
            alert('错误: ' + err.message);
        }
    };

    // 下载文件（需要 UserTask）
    const doDownloadFile = async (fileName: string) => {
        if (sn == null) return alert('请先连接设备');
        if (userTask == null) return alert('请先注册 UserTask');
        
        try {
            const jensen = mgr.getInstance(sn, userTask);
            if (jensen == null) return alert('设备未连接');
            
            await jensen.getFile(fileName, 1024, (data) => {
                if (data instanceof Uint8Array) {
                    console.log('文件数据:', data);
                    // 处理下载的数据...
                } else {
                    console.error('下载失败');
                }
            });
        } catch (err: any) {
            alert('错误: ' + err.message);
        }
    };

    return (
        <>
            <h1>设备管理器</h1>
            
            {/* 操作按钮 */}
            <div className="btn-container">
                <button onClick={doConnect}>连接设备</button>
                <button onClick={doRequestUserTask}>注册任务</button>
                <button onClick={doReleaseUserTask}>释放任务</button>
                <button onClick={doGetTime}>获取时间</button>
            </div>
            
            {/* 设备信息 */}
            <div>
                <h3>设备信息</h3>
                <ul>
                    <li>SN: {sn}</li>
                    <li>时间: {time}</li>
                    <li>电池: {battery}</li>
                    <li>录音中: {recording?.name || '无'}</li>
                </ul>
                
                <h3>文件列表</h3>
                <ul>
                    {files.map((file, index) => (
                        <li key={index}>
                            {file.name} - {file.length} bytes - {file.duration}ms
                            <a href="#" onClick={() => doDownloadFile(file.name)}>下载</a>
                        </li>
                    ))}
                </ul>
            </div>
        </>
    );
}
```

## API 参考

### ConnectionManager (mgr)

#### 连接管理

| 方法 | 说明 |
|------|------|
| `tryconnect()` | 尝试自动连接已授权的设备 |
| `connect()` | 打开设备选择对话框手动连接 |
| `close(sn)` | 关闭指定设备的连接 |
| `onautoconnect(handler)` | 注册自动连接成功的回调 |
| `onconnectionstatechanged(handler)` | 注册连接状态变化的回调 |

#### 任务管理

| 方法 | 说明 |
|------|------|
| `registerTask(sn, tag)` | 注册 UserTask（超时 3 秒） |
| `unregisterTask(sn, tag)` | 释放 UserTask |
| `keepUserTaskAlive(sn, tag)` | 保持 UserTask 活跃 |
| `verifyUserTask(sn, tag)` | 验证 UserTask 是否有效 |
| `getCurrentUserTaskTag(sn)` | 获取当前 UserTask 的标签 |
| `executeWithUserTask(sn, tag, callback)` | 验证后执行回调 |

#### 设备实例

| 方法 | 说明 |
|------|------|
| `getInstance(sn, userTaskTag)` | 获取受保护的 Jensen 实例 |

#### 状态管理

| 方法 | 说明 |
|------|------|
| `getDeviceState(sn, key)` | 获取设备状态值 |
| `setDeviceState(sn, key, value)` | 设置设备状态值 |
| `subscribeDeviceState(sn, key, listener)` | 订阅状态变更 |
| `isDeviceStateStale(sn, key, maxAge)` | 检查状态是否过期 |
| `dumpDeviceState(sn)` | 导出状态快照（调试用） |

### TaskManager

| 方法 | 说明 |
|------|------|
| `registerInitializeTask(task)` | 注册初始化任务 |
| `registerTimerTask(task)` | 注册定时任务 |
| `registerBackgroundTask(task)` | 注册后台任务 |
| `applyUserTask(tag, timeout)` | 申请 UserTask |
| `releaseUserTask(tag)` | 释放 UserTask |
| `verifyUserTask(tag)` | 验证 UserTask |

### DeviceStateStore

| 方法 | 说明 |
|------|------|
| `get(key)` | 获取状态值 |
| `set(key, value)` | 设置状态值（自动触发监听器） |
| `has(key)` | 检查状态是否存在 |
| `getUpdateTime(key)` | 获取最后更新时间 |
| `isStale(key, maxAge)` | 检查状态是否过期 |
| `subscribe(key, listener)` | 订阅状态变更（返回取消函数） |
| `clear()` | 清空所有状态 |
| `clearListeners()` | 清空所有监听器 |

## 常见问题

### 1. 为什么调用 `getInstance()` 报错？

确保传入了 `userTaskTag` 参数：
```typescript
// ❌ 错误
const jensen = mgr.getInstance(sn);

// ✅ 正确
const jensen = mgr.getInstance(sn, userTaskTag);
```

### 2. UserTask 被自动释放了？

UserTask 空闲超过 30 秒会被自动释放。对于长时间操作，定期调用：
```typescript
mgr.keepUserTaskAlive(sn, userTaskTag);
```

### 3. 如何知道设备断开了？

使用 `onconnectionstatechanged` 监听：
```typescript
mgr.onconnectionstatechanged((state, dinfo) => {
    if (state === 'disconnected') {
        console.log('设备断开:', dinfo);
    }
});
```

### 4. 状态订阅如何取消？

`subscribeDeviceState` 返回取消函数：
```typescript
const unsubscribe = mgr.subscribeDeviceState(sn, 'battery-status', listener);

// 取消订阅
unsubscribe?.();
```

## 开发调试

### 启动开发服务器

```bash
npm run dev
```

### 构建项目

```bash
npm run build
```

### 查看设备状态快照

```typescript
console.log(mgr.dumpDeviceState(sn));
```

---

**注意**：本库基于 WebUSB API，仅支持 Chrome 67+、Edge 79+、Opera 54+ 等现代浏览器。
