export type DeviceInfo = {
  sn: string;
  versionNumber: number;
  versionCode: string;
};

export type ReturnStruct = {
  common: { result: 'failed' | 'success' };
};

export type FileInfo = {
  name: string;
  createDate: string;
  createTime: string;
  time: Date;
  duration: number;
  length: number;
  signature: string;
};

export type BluetoothStatus = {
  status: string;
  mac?: string;
  name?: string;
  a2dp?: boolean;
  hfp?: boolean;
  avrcp?: boolean;
  battery?: number;
};

export type BluetoothDevice = {
  name: string;
  mac: string;
};

export const enum OS {
  Windows = 'Windows',
  Linux = 'Linux',
  Mac = 'Mac'
}

export const enum Platform {
  zoom = 'zoom',
  teams = 'teams',
  googleMeeting = 'google-meeting',
  webex = 'webex',
  feishu = 'feishu',
  lark = 'lark',
  wechat = 'wechat',
  line = 'line',
  whatsApp = 'whats-app',
  slack = 'slack',
  discord = 'discord'
}

export type ScheduleInfo = {
  startDate: Date;
  endDate: Date;
  os: OS;
  platform: Platform;
};

export enum Level {
  debug = 'debug',
  info = 'info',
  error = 'error'
}

export type Log = {
  level: Level;
  module: string;
  procedure: string;
  message: string;
  time: number;
};

export type Logger = {
  messages: Log[];
  consoleOutput: boolean;
  info(module: string, procedure: string, message: string): void;
  debug(module: string, procedure: string, message: string): void;
  error(module: string, procedure: string, message: string): void;
  _append(level: Level, module: string, procedure: string, message: string): void;
  _print(log: Log): void;
  filter(module: string, procedure: string): void;
  enableConsoleOutput(): void;
  disableConsoleOutput(): void;
  peek(rows: number): void;
  search(module: string, procedure: string, keyword: string): Log[];
};

declare class Jensen {
  constructor(log?: Logger);
  init: () => Promise<boolean>;
  connect: () => Promise<void>;
  onconnect: () => void;
  disconnect: () => void;
  ondisconnect?: () => void;
  isStopConnectionCheck: boolean;
  isConnected: () => boolean;
  getModel: () => string;
  tryconnect: (disableOnConnect?: boolean) => Promise<boolean>;

  getDeviceInfo: (time?: number) => Promise<DeviceInfo>;
  listFiles: (time?: number) => Promise<FileInfo[]>;
  getFile: (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void, onprogress?: (size: number) => void) => void;
  getFilePart: (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void, onprogress?: (size: number) => void) => void;
  getFileBlock: (fileName: string, length: number, on?: (msg: Uint8Array | 'fail') => void) => Promise<ReturnStruct['common']>;
  requestFirmwareUpgrade: (vn: number, length: number, time?: number) => Promise<{ result: 'accepted' | 'fail' }>;
  beginBNC: (time?: number) => Promise<ReturnStruct['common']>;
  endBNC: (time?: number) => Promise<ReturnStruct['common']>;
  setTime: (date: Date, timeout?: number) => Promise<ReturnStruct['common']>;
  deleteFile: (fileName: string) => Promise<{ result: 'failed' | 'success' | 'not-exists' }>;
  uploadFirmware: (data: number[], seconds?: number, onProgress?: (cur: number, total: number) => void) => Promise<ReturnStruct['common']>;
  getTime: (time?: number) => Promise<{ time: string }>;
  getSettings: (time?: number) => Promise<{ autoRecord: boolean; autoPlay: boolean; notification?: boolean } | null>;
  setAutoRecord: (enable: boolean, time?: number) => Promise<ReturnStruct['common']>;
  setAutoPlay: (enable: boolean, time?: number) => Promise<ReturnStruct['common']>;
  setNotification: (state: boolean, time?: number) => Promise<ReturnStruct['common']>;
  getRecordingFile: () => Promise<{ recording: null | string; createTime: string; createDate: string }>;
  getCardInfo: (seconds?: number) => Promise<{ used: number; capacity: number; status: string }>;
  formatCard: (seconds?: number) => Promise<ReturnStruct['common']>;
  factoryReset: (seconds?: number) => Promise<ReturnStruct['common']>;
  restoreFactorySettings: (seconds?: number) => Promise<ReturnStruct['common']>;
  getFileCount: (seconds?: number) => Promise<{ count: number } | null>;
  recordTestStart: (type: number, seconds?: number) => Promise<ReturnStruct['common']>;
  recordTestEnd: (type: number, seconds?: number) => Promise<ReturnStruct['common']>;
  test: (seconds?: number) => Promise<ReturnStruct['common']>;
  setBluetoothPromptPlay: (state: boolean, seconds?: number) => Promise<ReturnStruct['common']>;
  writeSerialNumber: (sn: string) => Promise<ReturnStruct['common']>;
  sendScheduleInfo: (info: ScheduleInfo[]) => Promise<ReturnStruct['common']>;
  scanDevices: (seconds?: number) => Promise<BluetoothDevice[]>;
  connectBTDevice: (mac: string, seconds?: number) => Promise<ReturnStruct['common']>;
  disconnectBTDevice: (seconds?: number) => Promise<ReturnStruct['common']>;
  getBluetoothStatus: (seconds?: number) => Promise<BluetoothStatus>;
  getRealtimeSettings: () => Promise<any>;
  startRealtime: () => Promise<ReturnStruct['common']>;
  pauseRealtime: () => Promise<ReturnStruct['common']>;
  stopRealtime: () => Promise<ReturnStruct['common']>;
  getRealtime: (frames: number) => Promise<{ data: Uint8Array; rest: number }>;
}

export = Jensen;
