import Jensen, { FileInfo } from "../../jensen";
import { Logger } from "../Logger";
import { TaskManager, DeviceStateStore } from "./tasks";

export type FileTaskData = {
    fileName: string;
    fileLength: number;
    fileSignature: string;
    task: 'download' | 'transcribe' | 'summarize';
}

/**
 * 任务构建器 - 注册设备的所有后台任务
 * @param jensen Jensen 实例
 * @param tm TaskManager 实例
 */
export function buildTasks(jensen: Jensen, tm: TaskManager) {
    // ===================== 初始化任务 =====================

    // 时间同步
    tm.registerInitializeTask({
        tag: 'time-sync',
        task: async (jensen: Jensen | null, store) => {
            let time = await jensen?.getTime(1);
            Logger.info('jensen', 'initialize', 'get-time: ' + JSON.stringify(time));
            
            // 存储设备时间到状态容器
            if (time) {
                store.set('time', { deviceTime: time.time, syncTime: new Date() });
            }
            
            // 同步设置时间
            await jensen?.setTime(new Date(), 1);
            Logger.info('jensen', 'initialize', 'set-time-to: ' + new Date().toLocaleString());
        }
    });

    // 获取设备设置
    tm.registerInitializeTask({
        tag: 'get-settings',
        task: async (jensen: Jensen | null, store) => {
            let settings = await jensen?.getSettings(1);
            Logger.info('jensen', 'initialize', 'get-settings: ' + JSON.stringify(settings));
            
            // 存储设置到状态容器
            if (settings) {
                store.set('settings', settings);
            }
        }
    });

    // 获取存储卡信息
    tm.registerInitializeTask({
        tag: 'get-card-info',
        task: async (jensen: Jensen | null, store) => {
            let cardInfo = await jensen?.getCardInfo(1);
            Logger.info('jensen', 'initialize', 'get-card-info: ' + JSON.stringify(cardInfo));
            
            // 存储卡信息到状态容器
            if (cardInfo) {
                store.set('cardInfo', cardInfo);
            }
        }
    });

    // 获取电池状态（仅 P1 型号）
    tm.registerInitializeTask({
        tag: 'battery-status',
        task: async (jensen: Jensen | null, store) => {
            if (jensen?.getModel() != 'hidock-p1') return;
            let batteryStatus = await jensen?.getBatteryStatus(1);
            Logger.info('jensen', 'initialize', 'get-battery-status: ' + JSON.stringify(batteryStatus));
            
            // 存储电池状态到状态容器
            if (batteryStatus) {
                store.set('battery-status', batteryStatus);
            }
        }
    });

    // ===================== 定时任务 =====================

    // 定时同步设备电池状态
    tm.registerTimerTask({
        tag: 'battery-status',
        interval: 1000,
        task: async (jensen: Jensen | null, store) => {
            if (jensen?.getModel() != 'hidock-p1') return;
            let batteryStatus = await jensen?.getBatteryStatus(1);
            Logger.info('jensen', 'timer-task', 'battery: ' + JSON.stringify(batteryStatus));
            
            // 更新电池状态（会自动触发订阅者）
            if (batteryStatus) {
                store.set('battery-status', batteryStatus);
            }
        }
    });

    // 定时获取设备时间
    tm.registerTimerTask({
        tag: 'get-device-time',
        interval: 1000,
        task: async (jensen: Jensen | null, store) => {
            if (jensen?.getModel() != 'hidock-p1') return;
            let time = await jensen?.getTime(1);
            Logger.info('jensen', 'timer-task', 'time: ' + JSON.stringify(time));
            
            // 更新设备时间
            if (time) {
                store.set('time', { deviceTime: time.time, syncTime: new Date() });
            }
        }
    });

    // 录音中的状态同步
    tm.registerTimerTask({
        tag: 'recording-status',
        interval: 1000,
        task: async (jensen: Jensen | null, store) => {
            let recording = await jensen?.getRecordingFile();
            Logger.info('jensen', 'timer-task', 'recording-status: ' + JSON.stringify(recording));
            if (recording) store.set('recording', recording);
        }
    });

    // 设备录音文件列表同步
    tm.registerTimerTask({
        tag: 'file-list-sync',
        interval: 1000,
        task: async (jensen: Jensen | null, store) => {
            let needRefresh = false;
            // 1. 定时查询文件数量
            let fileCount = await jensen?.getFileCount(1);
            if (fileCount)
            {
                let lastCount = store.get('file-count');
                if (lastCount != fileCount.count)
                {
                    needRefresh = true;
                    Logger.info('jensen', 'timer-task', 'file-count changed: ' + lastCount + ' -> ' + fileCount.count);
                }
                store.set('file-count', fileCount.count);
            }
            if (!needRefresh) return;
            let files = await jensen?.listFiles();
            if (files) store.set('files', files);
        }
    });

    // ===================== 后台任务 =====================
    // 如有需要可以在这里添加后台任务
    tm.registerBackgroundTask({
        tag: 'file-transfer',
        // 注意：必须使用 function 而非箭头函数，才能正确绑定 this
        schedule: async function (jensen, store) {
            let data = store.peekTaskData('file-transfer') as FileTaskData | undefined;
            if (!data) return;

            console.log('xxxx', data);

            // 通过 this 调用自定义方法
            await this.transfer(jensen, data, store);
            store.popTaskData('file-transfer');
            // this.postProcess(data);
            if (data.task == 'download')
            {
                this.download();
            }
            else if (data.task == 'transcribe')
            {
                this.transcribe();
            }
            else if (data.task == 'summarize')
            {
                this.summarize();
            }
        },
        // 自定义方法：文件传输
        transfer: async function (jensen: Jensen | null, data: FileTaskData, store: DeviceStateStore) {
            if (!jensen) return false;
            
            // 用 Promise 包装回调逻辑，等待所有数据块接收完成
            const fileData = await new Promise<Uint8Array | null>((resolve) => {
                const chunks: Uint8Array[] = [];
                let receivedLength = 0;

                jensen.getFile(data.fileName, data.fileLength, (chunk) => {
                    if (chunk === 'fail') {
                        console.error('文件传输失败');
                        resolve(null);
                        return;
                    }
                    
                    // 收集数据块
                    chunks.push(chunk);
                    receivedLength += chunk.length;
                    
                    console.log(`接收进度: ${receivedLength} / ${data.fileLength}`);
                    
                    // 检查是否接收完成
                    if (receivedLength >= data.fileLength) {
                        // 合并所有数据块为一个完整的 Uint8Array
                        const fileData = new Uint8Array(receivedLength);
                        let offset = 0;
                        for (const chunk of chunks) {
                            fileData.set(chunk, offset);
                            offset += chunk.length;
                        }
                        console.log('文件传输完成，总大小:', receivedLength);
                        resolve(fileData);
                    }
                });
            });
            
            if (fileData) {
                // 将完整的文件数据存储到 store 中供后续处理使用
                // store.set('transferredFile', { fileName: data.fileName, data: fileData });
                return true;
            }
            
            return false;
        },
        download: async function () {
            console.log('xxxx', 'download and save to local file');
        },
        transcribe: async function () {
            console.log('xxxx', 'transcribe the file');
        },
        summarize: async function () {
            console.log('xxxx', 'summarize the file');
        }
    });
}

