import { mgr } from './utils/mgr';
import './mgr.css';
import { useEffect, useState } from 'react';
import { Logger } from './Logger';
import Jensen from '../jensen';

export function DeviceManager() {

    const [sn, setSn] = useState<string|null>(null);
    const [time, setTime] = useState<string|null>(null);
    const [battery, setBattery] = useState<string|null>(null);
    const [userTask, setUserTask] = useState<string|null>(null);
    const [files, setFiles] = useState<Jensen.FileInfo[]>([]);
    const [recording, setRecording] = useState<Jensen.FileInfo|null>(null);

    useEffect(() => {
        mgr.onautoconnect((dinfo) => {
            setSn(dinfo.sn);
            // 注册一个状态监视器，当设备状态里的时间发生改变时，更新时间
            // 回调参数: (key, newValue, oldValue)
            mgr.subscribeDeviceState(dinfo.sn, 'time', (key, newValue, oldValue) => {
                // console.error('时间更新:', newValue);
                setTime(JSON.stringify(newValue));
            });
            mgr.subscribeDeviceState(dinfo.sn, 'battery-status', (key, newValue, oldValue) => {
                setBattery(JSON.stringify(newValue));
            });
            mgr.subscribeDeviceState(dinfo.sn, 'files', (key, newValue, oldValue) => {
                setFiles(newValue as Jensen.FileInfo[]);
            });
            mgr.subscribeDeviceState(dinfo.sn, 'recording', (key, newValue, oldValue) => {
                setRecording(newValue as Jensen.FileInfo | null);
            });
        });
    }, []);

    const doConnect = async () => {
        await mgr.tryconnect();
    }

    const doRequestUserTask = async () => {
        if (sn == null) return alert('Please connect first');
        const tid = new Date().getTime();
        setUserTask('t-' + tid);
        const rst = await mgr.registerTask(sn, 't-' + tid);
        if (!rst) return alert('Failed to request user task');
        Logger.info('mgr', 'task-manager', 'User task registered: t-' + tid);
    }

    const doReleaseUserTask = async () => {
        if (sn == null) return alert('Please connect first');
        if (userTask == null) return alert('Please request a user task first');
        await mgr.unregisterTask(sn, userTask);
        Logger.info('mgr', 'task-manager', 'User task released: ' + userTask);
        setUserTask(null);
    }

    const doGetTime = async () => {
        if (sn == null) return alert('Please connect first');
        if (userTask == null) return alert('Please request a user task first');
        
        try {
            // 方式一：使用 getInstance，现在必须传入 userTaskTag
            // 所有方法调用都会自动验证 UserTask
            // const jensen = mgr.getInstance(sn, userTask);
            // if (jensen == null) return alert('Please connect first');
            // const time = await jensen.getTime(1);
            // alert('Time: ' + time.time);
            
            // 方式二：使用 executeWithUserTask
            const time = await mgr.executeWithUserTask(sn, userTask, async (jensen) => {
                return await jensen.getTime(1);
            });
            alert('Time: ' + time.time);
        } catch (err: any) {
            alert('Error: ' + err.message);
        }
    }

    const doDownloadFile = async (file: Jensen.FileInfo) => {
        if (sn == null) return alert('Please connect first');
        try {
            // ...
            mgr.submitFileTask(sn, {
                fileName: file.name,
                fileLength: file.length,
                fileSignature: file.signature,
                task: 'download'
            });
        } catch (err: any) {
            alert('Error: ' + err.message);
        }
    }

    return (
        <>
            <h1>DeviceManager</h1>
            <div className="btn-container">
                <button onClick={doConnect}>Connect</button>
                <button onClick={doRequestUserTask}>Request</button>
                <button onClick={doReleaseUserTask}>Release</button>
                <button onClick={doGetTime}>Get Time</button>
            </div>
            <div>
                <h3>Device Info</h3>
                <ol>
                    <li>SN: {sn}</li>
                    <li>Time: {time?.toLocaleString()}</li>
                    <li>Battery: {battery}</li>
                    <li>Recording: {recording?.name}</li>
                </ol>
                <h3>Files</h3>
                <ol>
                {
                    files.map((file, index) => {
                        return <li key={index}>{file.name} - {file.length} - {file.duration} <a href="javascript:void(0)" onClick={() => { doDownloadFile(file) }}>Download</a></li>
                    })
                }
                </ol>
            </div>
        </>
    );
}